//**************************************************************************************
//*
//* filename: ImageProcessing.c
//* by: Zhaoyang Feng
//* at: Schafer Lab, UCSD
//*
//* Last modification date: 11/13/2001
//*
//**************************************************************************************

#include <ansi_c.h>
#include <userint.h>
#include <ImageProcessing.h>

//*Trying to move to seperated files.
void DilateOrErodeImage (unsigned char **image_matrix, int H, int W, int Dilate_nErode, int times, int bin_gray)
{
int row = 0;
int col = 0;
int p[10];
int sum_p = 0;
int tmp;
unsigned char **temp_image;
int i = 0,j; 

  	//* Create square matrix for storing temp image 
  	temp_image  = (unsigned char **) malloc (H * sizeof (unsigned char *));

  	if (!temp_image) 
  	{
    	printf("Out of memory\n");
    	exit(1);
  	}

  	for (i = 0; i < H; i++ ) 
  	{
    	temp_image[i] = (unsigned char *) malloc (W * sizeof (unsigned char ));
    	if (!temp_image[i]) 
    	{
      		printf("Out of memory\n");
      		exit(1);
    	}
  	}

  	//* Perform Dilation or Erosion <times> number of times 
  	for (i = 0; i < times; i++) 
  	{
 	   	//* Make a Copy of original image into temp image 
    	for (row = 0; row < H; row++) 
    	{ 
      		for(col = 0; col < W; col++ ) 
      		{  
        		temp_image[row][col]= image_matrix[row][col]; 
      		}
    	} 

    	//* Go through the entire image 
    	if (bin_gray == 0) 
    	{    //* binay image 
      		for (row = 0; row < H; row++) 
      		{
        		for (col = 0; col < W; col++) 
        		{
          		//* Get the pixel's neighbors and sum their values 
          			GetNeighbors (row, col, p, temp_image, H, W, 0);
          			sum_p=p[2]+p[3]+p[4]+p[5]+p[6]+p[7]+p[8]+p[9];

          			if (Dilate_nErode == 1) 
          			{    
          			//* Dilate Image 
 		        		//* If any neighbor is 1, the pixel is set to 1 (for dilation) 
		        		//* otherwise, there's no change in the pixel value  
            			if (sum_p >= 1) 
            			{
              				image_matrix[row][col] = 1;
            			}
            			else 
            			{
              				image_matrix[row][col] = temp_image[row][col];
            			}
          			}
          			else
          			{			
          			//* Erode Image 
       		    		//* If any neighbor is 0, the pixel is set to 0 (for erosion)
            			//* otherwise, there's no change in the pixel value 
            			if (sum_p < 8) 
            			{
              				image_matrix[row][col] = 0;
            			}
           				else 
           				{
              				image_matrix[row][col] = temp_image[row][col];
            			}
          			}
        		}
      		}
    	}
    	else if (bin_gray == 1) 
    	{    //* gray image 
      		for (row = 0; row < H; row++) 
      		{
        		for (col = 0; col < W; col++) 
        		{

          			//* Get the pixel's neighbors and sum their values 
          			GetNeighbors (row, col, p, temp_image, H, W, 255);

          			if (Dilate_nErode == 1) 
          			{    
          			//* Dilate Image 
            			tmp = p[1];
            			for (j = 2; j < 10; j++)
               				tmp = (tmp < p[j]) ? tmp : p[j];
            			image_matrix[row][col] = (unsigned char)tmp;
          			}
          			else
          			{                 
          			//* Erode Image 
            			tmp = p[1];
            			for (j = 2; j < 10; j++)
               				tmp = (tmp > p[j]) ? tmp : p[j];
            				image_matrix[row][col] = (unsigned char)tmp;
          			}
        		}
      		}
    	}
    	else 
    	{
      		printf("Invalid options for binary/gray image in Dilatation\n");
      		exit(1);
    	}

  	} //* end of for X times loop 


  	//* Delete temp images 
  	for (i = 0; i < H; i++ )
  	{
    	free(temp_image[i]);
  	}
  	free(temp_image);
}



void Smoothing (unsigned char **image_matrix, int H, int W, int times)
{
int row, col;
unsigned char **temp_image;
int i;
int p[10];
int sum_p;

//* Create square matrix for storing temp image 
  	temp_image  = (unsigned char **) malloc (H * sizeof (unsigned char *));

  	if (!temp_image) 
  	{
    	printf("Out of memory\n");
    	exit(1);
  	}

  	for (i = 0; i < H; i++ ) 
  	{
    	temp_image[i]    = (unsigned char *) malloc (W * sizeof (unsigned char ));
    	if (!temp_image[i]) 
    	{
      		printf("Out of memory\n");
      		exit(1);
    	}
  	}

	for (i = 0; i < times; i++)
  	{
    	//* Make a Copy of original image into temp image 
    	for (row = 0; row < H; row++)
      		for(col = 0; col < W; col++ )
        		temp_image[row][col]= image_matrix[row][col];

	    //* Go through the entire image 
    	for (row = 0; row < H; row++) 
    	{
      		for (col = 0; col < W; col++) 
      		{
        		//* Get the pixel's neighbors and sum their values 
        		GetNeighbors (row, col, p, temp_image, H, W, 255);
        		sum_p = p[1]+p[2]+p[3]+p[4]+p[5]+p[6]+p[7]+p[8]+p[9];

        		image_matrix[row][col] = (unsigned char)(sum_p / 9);
      		}
    	}
  	}  //*end of for X times loop 

  	//* Delete temp images 
  	for (i = 0; i < H; i++ )
  	{
    	free(temp_image[i]);
  	}

  	free(temp_image);
}


void ConvertToBinary(unsigned char **image_matrix, int H, int W, int BG_val)
{
int i, j;
  
  	for (i = 0; i < H; i++) 
  	{
    	for (j = 0; j < W; j++) 
    	{ 
      		if(image_matrix[i][j] ==  BG_val)
         		image_matrix[i][j] = 0;
      		else
         		image_matrix[i][j] = 1;
    	}
  	}
}




void GetNeighbors (int row, int col, int *p, unsigned char **image_matrix, int H, int W, int BG_val)
{
  //* Note BG_val is the background value, typically set to 0 for binary images 
  int pix;
  pix = image_matrix[row][col];
  p[1]= pix;

  if (row == 0)
  {
    if (col == 0)			//* Pixel in upper left corner 
    {
      p[2] = BG_val;
      p[3] = BG_val;
      p[4] = image_matrix[row][(col+1)];
      p[5] = image_matrix[(row+1)][(col+1)];
      p[6] = image_matrix[(row+1)][col];
      p[7] = BG_val; 
      p[8] = BG_val; 
      p[9] = BG_val; 
    }
    else if (col == (W-1))		//* Pixel in upper right corner  
    {
      p[2] = BG_val;
      p[3] = BG_val;
      p[4] = BG_val;
      p[5] = BG_val;
      p[6] = image_matrix[(row+1)][col];
      p[7] = image_matrix[(row+1)][(col-1)];
      p[8] = image_matrix[row][(col-1)];
      p[9] = BG_val; 
    }
    else				//* Pixel along top edge 
    {
      p[2] = BG_val;
      p[3] = BG_val;
      p[4] = image_matrix[row][(col+1)];
      p[5] = image_matrix[(row+1)][(col+1)];
      p[6] = image_matrix[(row+1)][col];
      p[7] = image_matrix[(row+1)][(col-1)];
      p[8] = image_matrix[row][(col-1)];
      p[9] = BG_val;
    }
  }
  else if (row == (H-1))
  {
    if (col == 0)			//* Pixel in lower left corner 
    {
      p[2] = image_matrix[(row-1)][col];
      p[3] = image_matrix[(row-1)][(col+1)];
      p[4] = image_matrix[row][(col+1)];
      p[5] = BG_val;
      p[6] = BG_val;
      p[7] = BG_val; 
      p[8] = BG_val; 
      p[9] = BG_val; 
    }
    else if (col == (W-1))		//* Pixel in lower right corner 
    {
      p[2] = image_matrix[(row-1)][col];
      p[3] = BG_val;
      p[4] = BG_val;
      p[5] = BG_val;
      p[6] = BG_val;
      p[7] = BG_val;
      p[8] = image_matrix[row][(col-1)];
      p[9] = image_matrix[(row-1)][(col-1)];
    }
    else				//* Pixel along bottom edge 
    {
      p[2] = image_matrix[(row-1)][col];
      p[3] = image_matrix[(row-1)][(col+1)];
      p[4] = image_matrix[row][(col+1)];
      p[5] = BG_val;
      p[6] = BG_val;
      p[7] = BG_val;
      p[8] = image_matrix[row][(col-1)];
      p[9] = image_matrix[(row-1)][(col-1)];
    }
  }
  else
  {
    if (col == 0)			//* Pixel along left edge 
    {
      p[2] = image_matrix[(row-1)][col];
      p[3] = image_matrix[(row-1)][(col+1)];
      p[4] = image_matrix[row][(col+1)];
      p[5] = image_matrix[(row+1)][(col+1)];
      p[6] = image_matrix[(row+1)][col];
      p[7] = BG_val; 
      p[8] = BG_val; 
      p[9] = BG_val; 
    }
    else if (col == (W-1))		//* Pixel along right edge 
    {
      p[2] = image_matrix[(row-1)][col];
      p[3] = BG_val;
      p[4] = BG_val;
      p[5] = BG_val;
      p[6] = image_matrix[(row+1)][col];
      p[7] = image_matrix[(row+1)][(col-1)];
      p[8] = image_matrix[row][(col-1)];
      p[9] = image_matrix[(row-1)][(col-1)];
    }
    else				//* Pixel is inside border 
    {
      p[2] = image_matrix[(row-1)][col];
      p[3] = image_matrix[(row-1)][(col+1)];
      p[4] = image_matrix[row][(col+1)];
      p[5] = image_matrix[(row+1)][(col+1)];
      p[6] = image_matrix[(row+1)][col];
      p[7] = image_matrix[(row+1)][(col-1)];
      p[8] = image_matrix[row][(col-1)];
      p[9] = image_matrix[(row-1)][(col-1)];
    }
  }
}

void GetNeighborsLabel (int row, int col, int *p, int **image_matrix, int H, int W, int BG_val)
{
  //* Note BG_val is the background value, typically set to 0 for binary images 
  int pix;
  pix = image_matrix[row][col];
  p[1]= pix;

  if (row == 0)
  {
    if (col == 0)			//* Pixel in upper left corner 
    {
      p[2] = BG_val;
      p[3] = BG_val;
      p[4] = image_matrix[row][(col+1)];
      p[5] = image_matrix[(row+1)][(col+1)];
      p[6] = image_matrix[(row+1)][col];
      p[7] = BG_val; 
      p[8] = BG_val; 
      p[9] = BG_val; 
    }
    else if (col == (W-1))		//* Pixel in upper right corner   
    {
      p[2] = BG_val;
      p[3] = BG_val;
      p[4] = BG_val;
      p[5] = BG_val;
      p[6] = image_matrix[(row+1)][col];
      p[7] = image_matrix[(row+1)][(col-1)];
      p[8] = image_matrix[row][(col-1)];
      p[9] = BG_val; 
    }
    else				//* Pixel along top edge 
    {
      p[2] = BG_val;
      p[3] = BG_val;
      p[4] = image_matrix[row][(col+1)];
      p[5] = image_matrix[(row+1)][(col+1)];
      p[6] = image_matrix[(row+1)][col];
      p[7] = image_matrix[(row+1)][(col-1)];
      p[8] = image_matrix[row][(col-1)];
      p[9] = BG_val;
    }
  }
  else if (row == (H-1))
  {
    if (col == 0)			//* Pixel in lower left corner 
    {
      p[2] = image_matrix[(row-1)][col];
      p[3] = image_matrix[(row-1)][(col+1)];
      p[4] = image_matrix[row][(col+1)];
      p[5] = BG_val;
      p[6] = BG_val;
      p[7] = BG_val; 
      p[8] = BG_val; 
      p[9] = BG_val; 
    }
    else if (col == (W-1))		//* Pixel in lower right corner 
    {
      p[2] = image_matrix[(row-1)][col];
      p[3] = BG_val;
      p[4] = BG_val;
      p[5] = BG_val;
      p[6] = BG_val;
      p[7] = BG_val;
      p[8] = image_matrix[row][(col-1)];
      p[9] = image_matrix[(row-1)][(col-1)];
    }
    else				//* Pixel along bottom edge 
    {
      p[2] = image_matrix[(row-1)][col];
      p[3] = image_matrix[(row-1)][(col+1)];
      p[4] = image_matrix[row][(col+1)];
      p[5] = BG_val;
      p[6] = BG_val;
      p[7] = BG_val;
      p[8] = image_matrix[row][(col-1)];
      p[9] = image_matrix[(row-1)][(col-1)];
    }
  }
  else
  {
    if (col == 0)			//* Pixel along left edge 
    {
      p[2] = image_matrix[(row-1)][col];
      p[3] = image_matrix[(row-1)][(col+1)];
      p[4] = image_matrix[row][(col+1)];
      p[5] = image_matrix[(row+1)][(col+1)];
      p[6] = image_matrix[(row+1)][col];
      p[7] = BG_val; 
      p[8] = BG_val; 
      p[9] = BG_val; 
    }
    else if (col == (W-1))		//* Pixel along right edge 
    {
      p[2] = image_matrix[(row-1)][col];
      p[3] = BG_val;
      p[4] = BG_val;
      p[5] = BG_val;
      p[6] = image_matrix[(row+1)][col];
      p[7] = image_matrix[(row+1)][(col-1)];
      p[8] = image_matrix[row][(col-1)];
      p[9] = image_matrix[(row-1)][(col-1)];
    }
    else				//* Pixel is inside border 
    {
      p[2] = image_matrix[(row-1)][col];
      p[3] = image_matrix[(row-1)][(col+1)];
      p[4] = image_matrix[row][(col+1)];
      p[5] = image_matrix[(row+1)][(col+1)];
      p[6] = image_matrix[(row+1)][col];
      p[7] = image_matrix[(row+1)][(col-1)];
      p[8] = image_matrix[row][(col-1)];
      p[9] = image_matrix[(row-1)][(col-1)];
    }
  }
}


//******************************BreakImage1DArrayTo2DArray***************************************************************
//
//		The function for displaying image by NI needs to breage 1D array to 2D array
//
//****************************************************************************************************************************
void BreakImage1DArrayTo2DArray (unsigned char *image, int imageHeight, int imageWidth, unsigned char **image_matrix)
{

int i, j;

	for (i = 0; i < imageHeight; i++)
	{
		for (j = 0; j < imageWidth; j++)
		{
			image_matrix[i][j] = image[i * imageWidth + j] ;
		}
	}
}


//******************************IntegerateImage2DArrayTo1DArray***************************************************************
//
//		The function for displaying image by NI needs to interate 2D to 1D array
//
//****************************************************************************************************************************
void IntegerateImage2DArrayTo1DArray (unsigned char **image_matrix, int imageHeight, int imageWidth, unsigned char *image)
{

int i, j;

	for (i = 0; i < imageHeight; i++)
	{
		for (j = 0; j < imageWidth; j++)
		{
			image[i* imageWidth + j] = image_matrix[i][j] ;
		}
	}
}

//******************************ThresholdingByHistogram***********************************************************************
//
//		This function will threshold a image maxtrix by histogram.  Above = 1, below = 0.
//
//****************************************************************************************************************************
void ThresholdingByHistogram(unsigned char **image_matrix, int H, int W, int BG_val)
{
  int i, j;
  int th_val;
  int histogram[256];
  int first_peak, second_peak;
  long max = 0;
  int avg, sum, num;

  //* Initialize the historgram array 
  for (i = 0; i < 256; i++)
    histogram[i] = 0;

  //* Compute the histogram 
  //* Processing only inside the original image area 
  for (i = 5; i < (H-5); i++)
    for (j = 5; j < (W-5); j++)
       histogram[(int)image_matrix[i][j]]++;

  //* Find the first peak by moving average 
  for (i = 0; i < 256; i++) {
    sum = 0;
    num = 0;
    for (j = -2; j < 2; j++) {
      if ((i+j) >= 0 && (i+j) < 256) {
        sum += histogram[i+j];
        num++; 
      }
    }
    avg = sum / num;

    if (avg > max) {
       first_peak = i;
       max = avg;
     }
  }
  

  //* Assume that second peak is at 0 
  second_peak = 0;
  
  //* Now, compute the threshold value 
  th_val = (first_peak + second_peak) / 2;

  if (BG_val == 255) { //* In case that back ground is white 
    for (i = 0; i < H; i++) {
      for (j = 0; j < W; j++) {
        if(image_matrix[i][j] < th_val)
           image_matrix[i][j] = 0;
        else
           image_matrix[i][j] = BG_val;
      }
    }
  }
  else { //* In case that back ground is black 
    for (i = 0; i < H; i++) {
      for (j = 0; j < W; j++) {
        if(image_matrix[i][j] < th_val)
           image_matrix[i][j] = BG_val;
        else
           image_matrix[i][j] = 255;
      }
    }
  }
}


//******************************RemoveIsolatedObjects*************************************************************************
//
//		This function will remove the smaller particle.
//
//****************************************************************************************************************************
void RemoveIsolatedObjects(unsigned char **image_matrix, int image_H, int image_W, int BG_val)
{
int row = 0;
int col = 0;
int p_label[10] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
int **labeled_image;
int *label_array;
int i = 0;
int num_objects = 0;
int num_pass = 0;
int change_val = TRUE;
int image_HW;
int min_val = 0;
int pix_label;
int label_val = 1;
int max_num_pixels=0, max_label;

	  //* Create square matrix for storing temp image of labels 
	labeled_image  = (int **) malloc (image_H * sizeof (int*));
	if (!labeled_image)
	{
	    MessagePopup ("Error!", "Can not allocate memory.");
	    exit(1);
	}
	  
	for (i = 0; i < image_H; i++ )
  	{
    	labeled_image[i]  = (int *) malloc (image_W * sizeof(int ));
    	if (!labeled_image[i])
    	{
	    	MessagePopup ("Error!", "Can not allocate memory.");
      		exit(1);
    	}

    	//* Initialize the labeled image array 
    	memset(labeled_image[i], 0, image_W);
  	}

  	image_HW = image_H * image_W;
  	label_array  = (int *) malloc ((image_HW+1) * sizeof(int));

  	//* Initialize the label array 
  	for (i = 0; i < (image_HW+1); i++)
    	label_array[i] = 0;

  	//* Initialize labeled_image by labelling all ON pixels with a unique label
  	//* And "turning ON" the label in the label array 
  	for (row = 0; row < image_H; row++)
  	{
    	for (col = 0; col < image_W; col++ )
    	{
      		if (image_matrix[row][col] != 0)
      		{
        		labeled_image[row][col] = label_val;
        		label_array[label_val]=1;
        		label_val++;
      		}
      		else
      		{
        		labeled_image[row][col] = 0;
        		label_array[label_val] = 0;
      		}
    	}
  	}

  	//* Iterate through image to find equivalent labels 
  	num_pass = 0;
  	change_val = TRUE;

  	while (change_val == TRUE)
  	{
    	num_pass++;
    	change_val = FALSE;

    	//* FIRST PASS: Top-down, left-to-right PASS 
    	for (row = 0; row < image_H; row++)
    	{
      		for(col = 0; col < image_W; col++ )
      		{
        		if (labeled_image[row][col] != 0)
        		{
          			GetNeighborsLabel (row, col, p_label,labeled_image, image_H, image_W, BG_val);

          			//* Find the smallest label in the neighbors 
          			min_val = p_label[1];
          			for (i = 2; i <= 9; i++)
          			{
            			if ((p_label[i] != 0) && (p_label[i] < min_val))
              				min_val = p_label[i];
          			}

          			//* Modify the labeled image with the smallest label 
          			pix_label=labeled_image[row][col];
          			if (pix_label != min_val)
          			{
            			label_array[pix_label]=0;  //* "delete" that label from list 
            			labeled_image[row][col] = min_val;
            			change_val = TRUE;
          			}
        		}
      		}
    	} //* end of first pass 

    	//* SECOND PASS: Bottom up, right-to-left PASS 
    	for (row = (image_H-1); row >= 0; row--)
    	{
      		for(col = (image_W-1); col >= 0; col-- )
      		{
        		if (labeled_image[row][col] != 0)
        		{
          			GetNeighborsLabel (row, col, p_label, labeled_image, image_H, image_W, BG_val);
          			//* Find the smallest label in the neighbors 
          			min_val = p_label[1];
          			for (i = 2; i <= 9; i++)
          			{
            			if ((p_label[i] != 0) && (p_label[i] < min_val))
              				min_val = p_label[i];
          			}

          			//* Modify the labeled image with the smallest label 
          			pix_label=labeled_image[row][col];
          			if (pix_label != min_val)
          			{
            			label_array[pix_label]=0;  //* "delete" that label from list 
            			labeled_image[row][col] = min_val;
            			change_val = TRUE;
          			}
        		}
      		}
    	} //* end of 2nd pass 
  	} //* end of while loop 


  	//* Count up the number of labeled objects 
  	num_objects = 0;
  	for (i = 1; i <= image_HW; i++)
  	{
    	if (label_array[i] != 0)
      	num_objects++;
  	}
  	
  	if (num_objects > 1) 
  	{

    	//* Initialize the label array for reuse 
    	for (i = 0; i < (image_HW+1); i++)
      		label_array[i] = 0;

    	for(row = 0; row < image_H; row++) 
    	{
      		for (col = 0; col < image_W; col++) 
      		{
         		pix_label = labeled_image[row][col];
         		
         		if (pix_label != 0) 
         		{
            		label_array[pix_label] = label_array[pix_label] + 1;
         		}
      		}
    	}

    	//* Find the label which has the largest number of pixels 
    	for (i = 1; i <= image_HW; i++) 
    	{
      		if (label_array[i] > max_num_pixels) 
      		{
         		max_num_pixels = label_array[i];
         		max_label = i;
      		}
    	}

    	for(row = 0; row < image_H; row++) 
    	{
      		for (col = 0; col < image_W; col++) 
      		{
        		if (labeled_image[row][col] != max_label) 
        		{
           			image_matrix[row][col] = 0;
        		}
      		}
    	}
  	}
  	
          
  	//* Delete labeled image 
  	for (i = 0; i < image_H; i++)
    	free(labeled_image[i]);
  	free(labeled_image);
  	free(label_array);

}


//******************************GenerateSkelPixelList*************************************************************************
//
//		This function will generize the list of the relative cooridate of a binary image where has white dot.
//				The return value is the total number fo the "white."
//
//****************************************************************************************************************************
int GenerateSkelPixelList(int * skelpixList_row, int * skelpixList_col, unsigned char **image_matrix, int imageHeight, int imageWidth)
{
int i,j;
int k=0;

  	for(j = 0; j < imageWidth; j++ )
  	{
  		for (i = 0; i < imageHeight; i++)
    	{
      		if(image_matrix[i][j] == 1 )              //* 1 means pixel is ON (white) 
      		{
        		skelpixList_row[k]=i;
        		skelpixList_col[k]=j;
        		k++;
      		} 
    	}
  	}

  	return k;
}


//******************************Binarization***************************************************************************************
//
//		This function will take a grey image char matrix and binarize it.  The retrun value is the area of the binarized image.
//
//*********************************************************************************************************************************
int Binarization(unsigned char  **cImageMatrix, int imageHeight, int imageWidth)
{
int i, j;
unsigned char **temp_image;
int new_height, new_width;
int BG_val;
int iArea = 0;

	//* Increase the image size by 10 pixels in row and column for morphology and then backgroud(255) padding  
	new_height = imageHeight + 10;
	new_width = imageWidth + 10;

  	temp_image = (unsigned char **) malloc (new_height * sizeof (unsigned char *));
  	if (!temp_image) {
	    MessagePopup ("Error!", "Can not allocate memory.");
    	exit(1);
  	}
  
  	for (i = 0; i < new_height; i++ ) 
  	{
    	temp_image[i] = (unsigned char *) malloc (new_width * sizeof (unsigned char ));

    	if (!temp_image[i]) 
    	{
		    MessagePopup ("Error!", "Can not allocate memory.");
      		exit(1);
    	}
    	memset(temp_image[i], 255, new_width);
  	}

  //* Copy original image into temp_image 
  	for (i = 0; i < imageHeight; i++)
  		for (j = 0; j < imageWidth; j++)
       		temp_image[i+5][j+5] = cImageMatrix[i][j];
  
  	//* Binarization 
  	BG_val = 255;
  	Smoothing(temp_image, new_height, new_width, 3);
  	ThresholdingByHistogram(temp_image, new_height, new_width, BG_val);
  	ConvertToBinary(temp_image, new_height, new_width, BG_val);
  	DilateOrErodeImage(temp_image, new_height, new_width, 1, 4, 0);
  	DilateOrErodeImage(temp_image, new_height, new_width, 0, 4, 0);
  	BG_val = 0;
  	RemoveIsolatedObjects(temp_image, new_height, new_width, BG_val);

  //* Restore the resulting image data into imageptr 
  	for (i = 0; i < imageHeight; i++)
  	{
    	for (j = 0; j < imageWidth; j++)
    	{
      		cImageMatrix[i][j] = temp_image[i+5][j+5];
      		//if (cImageMatrix[i][j] == 0) iArea ++;
      		if (cImageMatrix[i][j] != 0) iArea ++;
      	}
    }

  //* Free the allocated memory *
	for (i = 0; i < new_height; i++)
   	 	free(temp_image[i]);
  		free(temp_image);

  	return iArea; 
}

//******************************ThinImage*************************************************************************************
//
//		This function will thin a binary image down to a skeleton and retrun the length of this skeleton.
//
//****************************************************************************************************************************
int ThinImage ( unsigned char ** imageMatrix, int imageHeight, int imageWidth)
{
int row = 0;
int col = 0;
int i,k;
  
int sum_p = 0;
int prev_p = 0;
int transition_01=0;
  
int chk1 = 0;
int chk2 = 0;
int chk3_4 = 0;
  
int p_temp1 = 1;
int p_temp2 = 1;
  
int prev_length = 0;
int temp_length = 0;

int iteration = 0;
int subiteration;
int num_del = 1;
int total_del =0;
int p[10] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

unsigned char **temp_image;
int num_skelpix = 0;

int iLargerOne;
int * skelpixList_row, * skelpixList_col ;

  //* Create square matrix for storing temp image
  	temp_image = (unsigned char **) malloc (imageHeight * sizeof (unsigned char *));
 
  	if (!temp_image)
  	{
	    MessagePopup ("Error!", "Can not allocate memory.");
    	exit(1);
  	}

  	for (i = 0; i < imageHeight; i++ )
  	{
    	temp_image[i] = (unsigned char *) malloc (imageWidth * sizeof (unsigned char ));

    	memset(temp_image[i], 0, imageWidth);

    	if (!temp_image[i])
    	{
		    MessagePopup ("Error!", "Can not allocate memory.");
      		exit(1);
    	}
  	}
  	
  	//* Copy image into temp image 
  	for (row = 0; row < imageHeight; row++)
  	{ 
    	for(col = 0; col < imageWidth; col++ )
    	{  
      		temp_image[row][col]= imageMatrix[row][col]; 
    	}
  	} 

	iLargerOne = imageHeight * imageWidth ;

	skelpixList_row = (int *) malloc (iLargerOne * sizeof (int));
	skelpixList_col = (int *) malloc (iLargerOne * sizeof (int));

	num_skelpix = GenerateSkelPixelList(skelpixList_row, skelpixList_col, temp_image, imageHeight, imageWidth) ;

  //* Continue to iterate until no more deletes 
  	while (num_del !=0)
  	{
    	prev_length = num_skelpix;
    	temp_length = num_skelpix;
    	iteration += 1;
    	subiteration = iteration % 2;

    	//* Go through the object pixels & find/delete contour  
		for (i=0; i < num_skelpix; i++)
    	{
			row = skelpixList_row[i];
			col = skelpixList_col[i];

			GetNeighbors (row, col, p, imageMatrix, imageHeight, imageWidth, 0);
 
      		//* Check for condition 1 (sum of all neighbors) 
	    	sum_p = p[2]+p[3]+p[4]+p[5]+p[6]+p[7]+p[8]+p[9];
	    
      		if ((sum_p >= 2) && (sum_p <= 6))
        		chk1=1;
      		else
        		chk1=0;

      		//* Check for condition 2 (01 transitions of neighbors) 
      		prev_p=p[2];
      		transition_01=0;

      		for (k=3; k<=9; k++)
      		{
        		if (prev_p == 0 && p[k] == 1)
          		transition_01 += 1;
        		prev_p=p[k];
      		}

      		if (prev_p == 0 && p[2] == 1)
        		transition_01 += 1;
      
      		if (transition_01 == 1)
        		chk2=1;
      		else
        		chk2=0;

      		if (subiteration == 1) 
      		{   //* First Subiteration 
        		//* Check for condition 3 & 4 
        		//* This condition looks for points that are south or east boundary 
        		//* points or a north-west corner 
        		p_temp1=p[2]*p[4]*p[6];
        		p_temp2=p[4]*p[6]*p[8];
      		}
      		else 
      		{ 			//* Second Subiteration 
      			//* Note: These are different conditions than above
        		//* This condition looks for points that are north or west boundary
        		p_temp1=p[2]*p[4]*p[8];
        		p_temp2=p[2]*p[6]*p[8];
      		}

      		if ((p_temp1 == 0) && (p_temp2 == 0))
        		chk3_4=1;
      		else
        		chk3_4=0;

      		//* If all of the 4 conditions are met, then pixel is 
      		//* considered a part of the contour & can be deleted 
      		if ((chk1 == 1) && (chk2 == 1) && (chk3_4 == 1))
      		{
        		temp_image[row][col] = 0;	  //* Pixel turned OFF (= 0) 
        		temp_length -= 1;
      		}
      		else
      		{
        		temp_image[row][col] = imageMatrix[row][col];
      		}
    	}

    	//* Copy temp image into original image 
		for (row = 0; row < imageHeight; row++) 
    	{
			for (col = 0; col < imageWidth; col++) 
			{
        		imageMatrix[row][col] = temp_image[row][col];
      		}
   	 	}
   	 	
    	//* Generate skeleton pixel list & update skeleton length 
		num_skelpix = GenerateSkelPixelList(skelpixList_row, skelpixList_col, imageMatrix, imageHeight, imageWidth);
		num_del = prev_length - num_skelpix;
    	total_del += num_del;
  	} //* end of while loop
  	
  	//* Delete temp images 
  	for (i = 0; i < imageHeight; i++ )
  	{
    	free(temp_image[i]);
  	}

  	free(temp_image);
	free (skelpixList_row);
	free (skelpixList_col);

  	return num_skelpix; 
}


//******************************DelRedundantSkeleton**************************************************************************
//
//		This function further remove the redundent pixtel will thinned binary image skeleton.
//
//****************************************************************************************************************************
int DelRedundantSkeleton (unsigned char ** image_matrix, int imageWidth, int imageHeight)
{

int i, k;
int row, col;
int prev_p, transition_01;
int p[10] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

//JF 
int iLargerOne;
int num_skelpix = 0;
int * skelpixList_row, * skelpixList_col ;

	iLargerOne = imageHeight * imageWidth ;

	skelpixList_row = (int *) malloc (iLargerOne * sizeof (int));
	skelpixList_col = (int *) malloc (iLargerOne * sizeof (int));

	num_skelpix = GenerateSkelPixelList(skelpixList_row, skelpixList_col, image_matrix, imageHeight, imageWidth) ;


  /* Delete redundant pixels */
  	for (i = 0; i < num_skelpix; i++) 
  	{
    	row = skelpixList_row[i];
    	col = skelpixList_col[i];
	
		GetNeighbors (row, col, p, image_matrix, imageHeight, imageWidth, 0);

    	/* Check for 01 transitions of neighbors */
    	prev_p=p[2];
    	transition_01=0;
    	for (k=3; k<=9; k++)
    	{
      		if (prev_p == 0 && p[k] == 1)
          	transition_01 += 1;
      		prev_p=p[k];
    	}
    	
    	if (prev_p == 0 && p[2] == 1)
        transition_01 += 1;

    	if ((transition_01 < 3) && (p[2]*p[4] == 1 || p[4]*p[6] == 1 || p[6]*p[8] == 1 || p[8]*p[2] == 1)) 
    	{
      		image_matrix[row][col] = 0;
    	}
  	}
 
	num_skelpix = GenerateSkelPixelList(skelpixList_row, skelpixList_col, image_matrix, imageHeight, imageWidth) ;
	
	free (skelpixList_row);
	free (skelpixList_col);
	
	return num_skelpix;
}


