/*******Project Information*******************************************************************/
/*																							 */
/*		Worm Project:  Tracker system														 */
/*		This is one of the four parts of quantitative behavrior project (4/4)   			 */
/*																							 */
/*																							 */
/*		Zhaoyang Feng   																	 */
/*																							 */
/*		UCSD Biology, Schafer Lab															 */
/*																							 */
/*		Tracker group  (Zhaoyang Feng, Christopher Cronin, Paul Sternberg, William Schafer)	 */       
/*********************************************************************************************/
//******File Information**********************************************************************/
//**																						 */
//**  File name: 				backboneDataStruct.c										 */
//**  File function:			backbone data access functions      			 			 */
//**																						 */
//**  Caution:  This file and its h file are used by the second, third and forth parts of	 */
//**			quatitative behaviror project (2/4,3/4,4/4), convertor, lineup and   		 */
//**			miner.  Becareful when make changes											 */
//**																						 */
//**																						 */
//**  Author:					Zhaoyang (John) Feng										 */
//**							John Wittig (main structure is modified from his codes)		 */
//**  2nLast    modification:   March. 25, 2001.											 */
//**  Last time modification:   August, 19, 2003											 */
//**																						 */
//********************************************************************************************/
//******Update Information********************************************************************/
//**																						 */
//**		August, 2003  Christopher Cronin, Dr. Paul Sternberg from CalTech and Zhaoyang   */
//**             Feng, Dr.Schafer in UCSD decided to combine two systems together.			 */
//**              																			 */
//**		In this combined system, this data structure will serve as the joint point     	 */
//**		  of convertor/lineup (UCSD) and recognizor (CalTech)                            */
//**		                                                                                 */														 						 */
//**		wormDataStruct.c																 */
//**																						 */
//**																						 */
//**		Last modification data: 8/19/2003												 */
//********************************************************************************************/


//***************************codes start here


//#include <formatio.h>
#include <utility.h>
#include <ansi_c.h>
#include "lowlvlio.h"
#include "backboneDataStruct.h"  

//**************************************************************************************
//	BackboneNode constructor
//	
//  Given the initial values for the backbone node (see the data structure information in
//    backboneDataStruct.h file.), this function returns a backboneNode.
//
//  Usage: To use this constructor, just pass the data to this function and the function will
//			return a new backboneNode for usage
//**************************************************************************************
BackboneNode * newBackboneNode(int numXYpairs, double newTime, double newAbsCentX, double newAbsCentY, ImageInfor newImageInfor, double * newXpoints, double * newYpoints, int iVersion)
{
int index ; 
BackboneNode * newNode;

	newNode = (BackboneNode *) malloc ( sizeof(BackboneNode) ) ;	//memory allocation
	
	if (newNode == null) return MEM_ERROR ;
	
	//assigning values to a node
	newNode->Time = newTime;			  
	newNode->absCentX = newAbsCentX ;
	newNode->absCentY = newAbsCentY ;

	//If it is current version (needed to be dated and documented), assign value.
	if (iVersion <= 1)
	{
		newNode->imageInfor.iLoopType = newImageInfor.iLoopType;
		newNode->imageInfor.dArea = newImageInfor.dArea;
		newNode->imageInfor.dWormLength = newImageInfor.dWormLength;
		newNode->imageInfor.dTransparency = newImageInfor.dTransparency;
		newNode->imageInfor.dThickness = newImageInfor.dThickness;
		newNode->imageInfor.dFatness = newImageInfor.dFatness;

		//self define
		newNode->imageInfor.dLengthToPixelNumber = newImageInfor.dLengthToPixelNumber;
		//shape equivalence analysis
		newNode->imageInfor.dMaxIntercept = newImageInfor.dMaxIntercept;
		newNode->imageInfor.dMeanInterceptPerpendicular = newImageInfor.dMeanInterceptPerpendicular;
		newNode->imageInfor.dEquivalenceEllipsRatio = newImageInfor.dEquivalenceEllipsRatio;
		newNode->imageInfor.dEllipsMajorAxis = newImageInfor.dEllipsMajorAxis;
		newNode->imageInfor.dEllipsRatio = newImageInfor.dEllipsRatio;
		newNode->imageInfor.dRectBigSide = newImageInfor.dRectBigSide;
		newNode->imageInfor.dRectRatio = newImageInfor.dRectRatio;
		//shape feature

		newNode->imageInfor.dElongationFactor = newImageInfor.dElongationFactor;
		newNode->imageInfor.dCompactnessFactor = newImageInfor.dCompactnessFactor;
		newNode->imageInfor.dHeywoodCicularityFactor = newImageInfor.dHeywoodCicularityFactor;
		newNode->imageInfor.dTypeFactor = newImageInfor.dTypeFactor;
		newNode->imageInfor.dHydraulicRadius = newImageInfor.dHydraulicRadius;
		newNode->imageInfor.dWaddelDiskDiameter = newImageInfor.dWaddelDiskDiameter;
		newNode->imageInfor.dIXX = newImageInfor.dIXX;
		newNode->imageInfor.dIYY = newImageInfor.dIYY;
		newNode->imageInfor.dIXY = newImageInfor.dIXY;
//need to continew
	}
	
	//in case of add new data here as discussed in backboneDatastruct.h.
	if ( (newXpoints != null) && (newYpoints != null) )   //In condition that there is a valid skeleton points
	{
		// these point to memory allocated previously from the heap!!!
		newNode->xPoints = (double *) malloc( numXYpairs * sizeof(double) ) ;  //memeory allocation
		newNode->yPoints = (double *) malloc( numXYpairs * sizeof(double) ) ;
		newNode->iIsBkbp = 1;		//if a memory is allocated for skeleton points, set the flag
	
		for (index = 0; index < numXYpairs; index++)
		{
			newNode->xPoints[index] = newXpoints[index] ;		   //asign values
			newNode->yPoints[index] = newYpoints[index] ;
		}
	} else
	{
			newNode->xPoints = null ; //make sure it is null.
			newNode->yPoints = null ; //make sure it is null.
			newNode->iIsBkbp = 0;	  //Set the flag
	}

	return newNode ;
}

//**************************************************************************************
//	BackboneNode destructor
//	
//  Free the memory of a node
//
//  Usage: To use this destructor, just pass the node (pointer) needs to be destructed.
//			return a flag
//**************************************************************************************
int destroyBackboneNode(BackboneNode * thisNode)
{
	// possibly the node was not initialized, should not be deleting it though
	if (!thisNode) return 0 ;
	
	if (!thisNode->xPoints) free(thisNode->xPoints) ;
	if (!thisNode->yPoints) free(thisNode->yPoints) ;
	
	free(thisNode) ;
	
	return 1 ;
}

//**************************************************************************************
// readBackboneNode
//
// Usage: Pass the bockboneSet pointer and the position in the file where current node located
//        function will return a backboneNode
//
// **************************************************************************************
BackboneNode * readBackboneNode(int filePtr, BackboneSet * thisSet, int iVersion)
{
  
double newTime;
double newAbsCentX, newAbsCentY;
double * newXpoints, * newYpoints ;
ImageInfor newImageInfor;
int iIsbbp;

	//The data is saved and read one by one
	read (filePtr, &newTime, 8);      
	read (filePtr, &newAbsCentX, 8); 
	read (filePtr, &newAbsCentY, 8); 
	
	//Actually this is mean current version which is dated before 8/19/2003.  The new version is an issue only when mentioned.
	if (iVersion <= 1)
	{
		read (filePtr, &newImageInfor.iLoopType, 4); 
		read (filePtr, &newImageInfor.dArea, 8);

		//The error control kind of sucks, but do not want to spend a lot of time.
		//Frankly speaking, I can not remember how I exacetly did this.  However, it works.  
		//The error control is likely not modified later.  So, keep it as it is.
		if ( (newTime == - 2.0))
		{
			if( (newImageInfor.dArea == -2.0) && (newAbsCentX == -2.0) && (newAbsCentY == -2.0) )
			{
				if( (newImageInfor.iLoopType == -2) || (newImageInfor.iLoopType == -2) )

				return null;
			}
		}

		//read data one by one
		read (filePtr, &newImageInfor.dWormLength, 8); 
		read (filePtr, &newImageInfor.dTransparency, 8); 
		read (filePtr, &newImageInfor.dThickness, 8);
		read (filePtr, &newImageInfor.dFatness, 8); 
		read (filePtr, &newImageInfor.dLengthToPixelNumber, 8);
		read (filePtr, &newImageInfor.dMaxIntercept, 8); 
		read (filePtr, &newImageInfor.dMeanInterceptPerpendicular, 8); 
		read (filePtr, &newImageInfor.dEquivalenceEllipsRatio, 8); 
		read (filePtr, &newImageInfor.dEllipsMajorAxis, 8); 
		read (filePtr, &newImageInfor.dEllipsRatio, 8); 
		read (filePtr, &newImageInfor.dRectBigSide, 8); 
		read (filePtr, &newImageInfor.dRectRatio, 8); 

		read (filePtr, &newImageInfor.dElongationFactor, 8); 
		read (filePtr, &newImageInfor.dCompactnessFactor, 8); 
		read (filePtr, &newImageInfor.dHeywoodCicularityFactor, 8); 
		read (filePtr, &newImageInfor.dTypeFactor, 8); 
		read (filePtr, &newImageInfor.dHydraulicRadius, 8); 
		read (filePtr, &newImageInfor.dWaddelDiskDiameter, 8); 
		read (filePtr, &newImageInfor.dIXX, 8); 
		read (filePtr, &newImageInfor.dIYY, 8); 
		read (filePtr, &newImageInfor.dIXY, 8); 
		
	}
	
	read (filePtr, &iIsbbp, 4);

	//read data only if there is a skeleton pont set
	if (iIsbbp == 1)
	{
		newXpoints = (double *) calloc (thisSet->NumXYpairs, sizeof(double) ) ;
		newYpoints = (double *) calloc (thisSet->NumXYpairs, sizeof(double) ) ;

		read (filePtr, newXpoints, 8 * thisSet->NumXYpairs) ;
		read (filePtr, newYpoints, 8 * thisSet->NumXYpairs) ;
	} else
	{
	  	newXpoints = null;			   //make sure it is null.
	  	newYpoints = null;			   //make sure it is null.
	
	}
	
	//call constructor and pass the data 																	
	return newBackboneNode(thisSet->NumXYpairs, newTime, newAbsCentX, newAbsCentY, newImageInfor, newXpoints, newYpoints, iVersion); 
}

//**************************************************************************************
//				BackboneNode:: writeBackboneNode
//**************************************************************************************
int writeBackboneNode(BackboneNode * thisNode, int filePtr, BackboneSet * thisSet)
{
int error ;
double negTwo = -2.0;
int iZero = 0;
int iOne = 1;
int iMinusTwo = -2;

	//I really do not know why I used such stupid error control.  Just keep as it is.	
	if (thisNode == null) // in case error, i.e. invalid node from a damaged/deleted image  
	{
		// if a null node, write the height and width are zero, and move onto the next node
		write (filePtr, &negTwo, 8);    //Time
		write (filePtr, &negTwo, 8);   //absCentX
		write (filePtr, &negTwo, 8);   //absCentY
		
		//
		write (filePtr, &iMinusTwo, 4);	//iLoopTyp
		write (filePtr, &negTwo, 8);	//dArea
		
		return 0 ;	  //never used to control error though.
	}

	//Write data one by one.
	write (filePtr, &(thisNode->Time), 8);
	write (filePtr, &(thisNode->absCentX), 8);
	write (filePtr, &(thisNode->absCentY), 8);

	write (filePtr, &thisNode->imageInfor.iLoopType, 4); 
	write (filePtr, &thisNode->imageInfor.dArea, 8); 
	write (filePtr, &thisNode->imageInfor.dWormLength, 8); 
	write (filePtr, &thisNode->imageInfor.dTransparency, 8); 
	write (filePtr, &thisNode->imageInfor.dThickness, 8); 
	write (filePtr, &thisNode->imageInfor.dFatness, 8); 
	write (filePtr, &thisNode->imageInfor.dLengthToPixelNumber, 8);
	write (filePtr, &thisNode->imageInfor.dMaxIntercept, 8); 
	write (filePtr, &thisNode->imageInfor.dMeanInterceptPerpendicular, 8); 
	write (filePtr, &thisNode->imageInfor.dEquivalenceEllipsRatio, 8); 
	write (filePtr, &thisNode->imageInfor.dEllipsMajorAxis, 8); 
	write (filePtr, &thisNode->imageInfor.dEllipsRatio, 8); 
	write (filePtr, &thisNode->imageInfor.dRectBigSide, 8); 
	write (filePtr, &thisNode->imageInfor.dRectRatio, 8); 
	write (filePtr, &thisNode->imageInfor.dElongationFactor, 8); 
	write (filePtr, &thisNode->imageInfor.dCompactnessFactor, 8); 
	write (filePtr, &thisNode->imageInfor.dHeywoodCicularityFactor, 8); 
	write (filePtr, &thisNode->imageInfor.dTypeFactor, 8); 
	write (filePtr, &thisNode->imageInfor.dHydraulicRadius, 8); 
	write (filePtr, &thisNode->imageInfor.dWaddelDiskDiameter, 8); 
	write (filePtr, &thisNode->imageInfor.dIXX, 8); 
	write (filePtr, &thisNode->imageInfor.dIYY, 8); 
	write (filePtr, &thisNode->imageInfor.dIXY, 8); 
	
	if ( ( (thisNode->xPoints != NULL) && (thisNode->yPoints != NULL ) ) || (thisNode->iIsBkbp == 1) )
	{
		write (filePtr, &iOne, 4); 
	}else 
	{
		write (filePtr, &iZero, 4); 
	}
	
	if ( ( (thisNode->xPoints != NULL) && (thisNode->yPoints != NULL ) ) || (thisNode->iIsBkbp == 1) )
	{
		error = write (filePtr, thisNode->xPoints, 8 * thisSet->NumXYpairs) ;
		// Seems I wanted to add an error control but did not finished.  It is never a problem so, keep as it is.
		if (error != 8 * thisSet->NumXYpairs) ;   //This basicly said, if the byte write is incorrect
		error = write (filePtr, thisNode->yPoints, 8 * thisSet->NumXYpairs) ;
	}

	return 1; 

}


// **************************************************************************************
//	BackboneSet constructor, pass the file name, the number of skeleton point in a skeleton point set, 
//		the number of the total node;
// **************************************************************************************
BackboneSet * newBackboneSet(char * newBackboneSetName, int newNumXYpairs, int newNodeCount)
{

int index ;
BackboneSet * newSet ;
	
	//claim the memory
	newSet = (BackboneSet *) malloc ( sizeof(BackboneSet) ) ;

//setName
	
	//This is John Wittig's odd way to get read of file extention name.  However, there always is a 0 added to the file name.
	//That is the only reason I keep it here at that time.
	//In the future, the image data extention name may be changed to mov or wrm (stands for worm). 
	//In that case it, will be rewritten anyway.  This is reason I will not modify it at this time.
	
	index = 0 ;
	while (newBackboneSetName[index] != '\0' && index < MAX_NAME_LEN) index++ ;
	index-- ;
	newBackboneSetName[index--] = '\0' ;	 // 't'
	newBackboneSetName[index--] = '\0' ;	 // 'a'
	newBackboneSetName[index--] = '\0' ;	 // 'd'
	newBackboneSetName[index--] = '\0' ;	 // '.'

	newSet->setName = (char *) malloc ( (index + 10) * sizeof(char) ) ;
	sprintf(newSet->setName, "%sBack.bkb", newBackboneSetName); //later need to get read of Back.
	//sprintf(newSet->setName, "%s.bkb", newBackboneSetName);

//setDate for version control
	strcpy( newSet->Date, DateStr() ) ;    //Past the date of bkb data produced to data
	
	newSet->isBbpAligned = - 1;			  //The new backbone is not aligned.
	newSet->NumXYpairs = newNumXYpairs ;  //
	newSet->NodeCount = newNodeCount ;
	
//other count should be modified later
	newSet->filePtr = null ;
	
	// allocate space to point to all of the BackboneNodes within (pointers are size long)
	newSet->backboneNodePtr = (BackboneNode **) malloc ( newSet->NodeCount * sizeof(BackboneNodePtr) ) ;

	for (index = 0; index < newSet->NodeCount; index++)
	{
		// make sure the pointers of node aren't pointing anywhere.  This is an error control
		newSet->backboneNodePtr[index] = null ;
	}
	
	return newSet ;	   //return the set.
}


// **************************************************************************************
//	ReadBackboneSet
//		read backboneSet from a file.  The name of the BackboneSet which is actually the file name excluding
//			file extension name.
//	Usage: just pass the backboneSet name to the function.
// **************************************************************************************
BackboneSet * readBackboneSet(char * newBackboneSetName)
{

int index, error; 

int testFilePtr ;
BackboneSet * newSet ;
char fileName[MAX_NAME_LEN + 3] ; 
char charCount[4] ;
int iVersion = -1;

    //open a file with given name for read only
	testFilePtr = open (newBackboneSetName, O_RDONLY, 0);
	
	if ( testFilePtr == -1 ) return null ;
	
	//claim memory
	newSet = (BackboneSet *) malloc ( sizeof(BackboneSet) ) ;

	//pass the pointer to the set  and go to position zero
	newSet->filePtr = testFilePtr ;
	lseek (newSet->filePtr, 0, SEEK_SET);

//read name	
	index = 0 ;
	//again this is John Wittig's odd way to remove the extertion name.
	//Keep as it is for this time.
	//In the future it will be rewritten anyway.
	while (newBackboneSetName[index] != '\0' && index < MAX_NAME_LEN + 3) index++ ;
	newSet->setName = (char *) malloc ( (index + 1) * sizeof(char) ) ;
	strcpy(newSet->setName, newBackboneSetName) ; 

//read date and get the version control here
	error = read (newSet->filePtr, &(newSet->Date), 11);
	getBackboneVersion (newSet, &iVersion);

//read data one by one
	error = read (newSet->filePtr, &(newSet->isBbpAligned), 4);
	error = read (newSet->filePtr, &(newSet->NumXYpairs), 4);
	error = read (newSet->filePtr, &(newSet->NodeCount), 4);
	error = read (newSet->filePtr, &(newSet->type1LoopCount), 4);
	error = read (newSet->filePtr, &(newSet->type2LoopCount), 4);
	error = read (newSet->filePtr, &(newSet->type3LoopCount), 4);
	error = read (newSet->filePtr, &(newSet->invalidNodeCount), 4);

	// allocate space to point to all of the BackboneNodes within (pointers are size long)
	newSet->backboneNodePtr = (BackboneNode **) malloc ( newSet->NodeCount * sizeof(BackboneNodePtr) ) ;

    //read Node one by one
	for (index = 0; index < newSet->NodeCount; index++)
	{
		newSet->backboneNodePtr[index] = readBackboneNode(newSet->filePtr, newSet, iVersion) ;
	}

	return newSet ;
}


//**************************************************************************************
//  writeBackboneSet
//    write a backboneSet to file
// 
//  Usage: just pass the backboneSet to the function.
//  
//**************************************************************************************
int writeBackboneSet(BackboneSet * thisBackboneSet)
{

int index, error ;
char fileName[MAX_NAME_LEN + 3] ;
	
	// create new file, or erase current file, position the file pointer to the beginning of the file
	thisBackboneSet->filePtr = open (thisBackboneSet->setName, O_WRONLY | O_CREAT | O_TRUNC, S_IRWXU);

	// write the startDate and startTime strings, followed by the number of Backbones in the Set
	error = write (thisBackboneSet->filePtr, &(thisBackboneSet->Date), 11);
	error = write (thisBackboneSet->filePtr, &(thisBackboneSet->isBbpAligned), 4);
	error = write (thisBackboneSet->filePtr, &(thisBackboneSet->NumXYpairs), 4);
	error = write (thisBackboneSet->filePtr, &(thisBackboneSet->NodeCount), 4);
	error = write (thisBackboneSet->filePtr, &(thisBackboneSet->type1LoopCount), 4);
	error = write (thisBackboneSet->filePtr, &(thisBackboneSet->type2LoopCount), 4);
	error = write (thisBackboneSet->filePtr, &(thisBackboneSet->type3LoopCount), 4);
	
	error = write (thisBackboneSet->filePtr, &(thisBackboneSet->invalidNodeCount), 4);

	for (index = 0; index < thisBackboneSet->NodeCount; index++)
	{
		writeBackboneNode(thisBackboneSet->backboneNodePtr[index], thisBackboneSet->filePtr, thisBackboneSet) ;
	}
	
	return close (thisBackboneSet->filePtr) ;	   // -1 is error, all else is okay
}


// **************************************************************************************//
//	BackboneSet destructor
//    Pretty much say the name
// **************************************************************************************//
int destroyBackboneSet(BackboneSet ** BackboneSetPtrPtr)
{
	int index ;

	BackboneSet * thisSet = *BackboneSetPtrPtr ;
	
	// catch the NULL pointer 
	if (thisSet == null) return (0) ;
	
	free( thisSet->setName ) ;
	
	for (index = 0; index < thisSet->NodeCount; index++)
	{
		destroyBackboneNode( thisSet->backboneNodePtr[index] ) ;
		thisSet->backboneNodePtr[index] = null ;
	}
	
	free ( thisSet->backboneNodePtr ) ;
	
	// can free the memory through another pointer, but need to set this to zero
	// to effect thisBackboneSet in main
	*BackboneSetPtrPtr = null ;
	
	return 1 ;
}	


// **************************************************************************************************//
//	addBackbone
//  if there is a valid node pass in which indicated by a positive size of worm; a new node is added 
//		to the backboe set and backbone node constructor will be called. 
// **************************************************************************************************//
int addBackbone(BackboneSet * thisBackboneSet, int whichNode, double newTime, double newAbsCentX, double newAbsCentY, ImageInfor newImageInfor, double * newXpoints, double * newYpoints, int iVersion)
{

	//if an Backbone is passed in, create an Backbone node to hold it, otherwise, point to NULL
	if (newImageInfor.dArea >= 0)																							
	{																														
		thisBackboneSet->backboneNodePtr[whichNode] = newBackboneNode (thisBackboneSet->NumXYpairs, newTime, newAbsCentX, newAbsCentY, newImageInfor, newXpoints, newYpoints, iVersion) ;
	}
	else  thisBackboneSet->backboneNodePtr[whichNode] = null ;
	
	if (thisBackboneSet->backboneNodePtr[whichNode] == null && newXpoints != null && newYpoints != null) return MEM_ERROR ;
	
	return 1 ;
}

//********************************************************************************//
//  getBackboneVersion
//
//	Version control.  The date of the data is intrisically saved.  
//
//  Pass a backboneSet to it; and the iVersion will be modified.
//********************************************************************************//
void getBackboneVersion (BackboneSet *newSet, int *iVersion)
{
int iYear;
char sDate[9] = "";

	sDate[0] = newSet->Date[6];
	sDate[1] = newSet->Date[7];
	sDate[2] = newSet->Date[8];
	sDate[3] = newSet->Date[9];
	sDate[4] = newSet->Date[0];
	sDate[5] = newSet->Date[1];
	sDate[6] = newSet->Date[3];
	sDate[7] = newSet->Date[4];

	//If new data is added to the data set, just add versions here.
	iYear = strtoul (sDate, NULL, 10);
	if ( iYear >= 20011210)
	{
		*iVersion = 1;
	} 

}
