#include <userint.h>
#include <ansi_c.h>
#include "backBoneAnalysis.h"

// ********************************************************************************************** //
// ********************************************* CONSTANTS ************************************** //
// ********************************************************************************************** //

#define N_DIR  1
#define NE_DIR 2
#define E_DIR  3
#define SE_DIR 4
#define S_DIR  5
#define SW_DIR 6
#define W_DIR  7
#define NW_DIR 8
#define NUM_DIRS 8 // how many directions are there, 4 cardinal and 4 inbetween
#define NO_POI   0 // if there is no point of interest in a direction, this is the return value

#define dimOfNeighborHood = 3 // we want our neighborhoods to be 3 by 3 arrays of pixels

#define GEN_FAIL  0
#define SUCCESS   1

#ifndef TRUE
	#define TRUE 1
#endif
#ifndef FALSE
	#define FALSE 0
#endif
#ifndef null
	#define null 0
#endif



#define PAD 1
#define BACKGROUND 0
#define FOREGROUND 1

#define DIAG_DIST 1.41421356237 // sqrt(2)
#define STRAIGHT_DIST 1
#define END_POINT 1
#define NOTHING_IN_THIS_DIR -1

// ********************************************************************************************** //
// ********************************************* STRUCTURES ************************************* //
// ********************************************************************************************** //

// using memcopy, if any pointers are ever used in the data structure
// REMEMBER TO REWORK THE MEMCOPY PORTION OF THE CODE (in passCurrInfoToPrev)

typedef struct iPoints
{
	int x;
	int y;
} iPoint;

typedef struct dPoints
{
	double x;
	double y;
} dPoint;

typedef struct PointOfInterests // (from here on PointsOfInterest will be "PsOI" and pointOfInterest will be "poi")
{
	iPoint coord; // list of x and y coordinates of the poi (x: positive is right, y: positive is down. starting at top left)
	int numNeighs; // the number (out of 8 theoretically and 4 in this case) of on pixels in the poi's vicinity/neighborhood
	int isInLoop;  // boolean: is this poi part of a backbone loop?
	// these next two are a little tricky.  Here's an example: 
		// Which example - if I want to know if there's a neighbor to the East, I will say thisPOI.which[E_DIR], and it will 
		// return the index into the list of poi's that you would reach if your initial direction of travel were east. 
		// Dist example - if I want to know how far the neighbor to the NE is, thisPOI.dist[NE_DIR].  
		// Both return NO_POI if there isn't one in that direction.
	int which[NUM_DIRS+1]; // holds the # of the poi in the direction of the index (N=1, NE=2, E=3 etc.)
	int rev[NUM_DIRS+1]; // holds the direction that a poi-neighbor in a certain direction (the index) must first go to reach this poi
	double dist[NUM_DIRS+1]; // holds the distance to the poi in the direction of the index (N=1, NE=2 etc.)
} PointOfInterest;

typedef struct maps
{
	char inst[(int)MAX_NUM_PSOI/2+2]; // instructions
	int numInsts; // also serves as an indicator (when looping only) as whether or not there is a map to this POI
} Map;

typedef struct Tails
{
	Map map[(int)MAX_NUM_PSOI/2+2]; // map to all poi's sticking out from loop (the tail)
	double dists[(int)MAX_NUM_PSOI/2+2]; // the distances to each poi
	double greatestWeight;
	int poiGreatestWeight;
} Tail;

typedef struct Backbones
{
	double dists[MAX_PIXELS_IN_WORM-1];
	int numPix;
	iPoint coord[MAX_PIXELS_IN_WORM];
	dPoint sample[MAX_PIXELS_TO_REP_BACKBONE];
	int numSamples;
	
} Backbone;

typedef struct AnalyzedBackBoneInfos
{
	PointOfInterest PsOI[MAX_NUM_PSOI]; 
	int numPsOI;          
	int isWormInLoop; 
	int headPOI;
	int tailPOI;
	int intersectPOI;
	int LD;
	Map mapLD;
	Tail tail;
	double contBackbone[MAX_PIXELS_IN_WORM];
	Backbone BB;
	
} AnalyzedBackboneInfo;





// ********************************************************************************************** //
// ********************************************* GLOBALS **************************************** //
// ********************************************************************************************** //

AnalyzedBackboneInfo gPrevInfo;
AnalyzedBackboneInfo gInfo;

int gIsPrevInfoValid = FALSE;
int gSuccess;

#ifdef DEBUG
	int gWindowExists[10];
#endif


	
	
// ********************************************************************************************** //
// ******************************** "PRIVATE" FUNCTION PROTOTYPES ******************************* //
// ********************************************************************************************** //

 
int initializeInfo( void );
int findPointsOfInterest(char ** binImg,int height,int width, PointOfInterest * PsOI,int * numPsOI);



int padImg(char ** img, int height, int width, int padWidth); 
int makeVicinity(char ** img, int y, int x, char * vicinity);
int exploreDir( char ** img, PointOfInterest * startPoi, int firstMove, int * numPsOI, PointOfInterest * PsOI, int height, int width );      
int thinCondition( char cond, char * P );
int isPoi(char * vicinity, int  * numNeighs);
int distFromHereToEverywhere( int startPOI, PointOfInterest * PsOI, double * dists, char * restrictedDir, Map * map, int * isWormInLoop, char ** img, int height, int width );
int maxDist(char ** img, int height, int width);     
int	pruneImage(char ** img, int height, int width, AnalyzedBackboneInfo * info, int loopPOI, int dirToIgnore); 
int moveForward( double * totalDist, int * cameFrom, int * y, int * x, char * vicinity );  
int move( int dirToMove, int * cameFrom, int * y, int * x );  
int deleteAllTouching(char ** img, int y, int x, int dontDeleteY, int dontDeleteX);
int resetCurrInfo(void);
int findLoopNodeWithHeaviestBranch(char ** img, int height, int width) ;
int sampleSkel( char ** img, int height, int width, double * backboneY, double * backboneX, PointOfInterest * headPOI, PointOfInterest * tailPOI, int numPixToRepBackbone);
int passCurrInfoToPrev(void);
int switchHeadAndTailIfNecessary(void);
void showImageWithGrayPix( char ** char2d, int height, int width, int windowNumber, dPoint * pointGray, int numPix, char grayVal );
int resetExtractBackboneFromBinImg(void); 
 
// ********************************************************************************************** //
// ***************************************** MAIN FUNCTIONS ************************************* //
// ********************************************************************************************** //

// **************************** extractBackboneFromBinImg **************************************** //          
int extractBackboneFromBinImg(  
								// in
								char ** binImg, // binary image to extract backbone from 
		   						int numPixToRepBackbone,
		   						int width,
		   						int height,
		   						//out
		   						double * backboneX,
			   					double * backboneY 
								)
{
	int index = 0;

	#ifdef DEBUG
		for( index = 0; index<9; index++) 
		{			
			gWindowExists[index] = 0;
		}
		showImage(binImg, height, width, 1);
	#endif
	
	
	gSuccess = findPointsOfInterest( binImg, height, width, gInfo.PsOI, &gInfo.numPsOI );
		if (gSuccess==FALSE) 
		{
			return 0;   			
		}
	//#ifdef DEBUG
	//	showImage(binImg, height, width, 1);
	//#endif	

	gSuccess = maxDist(binImg, height, width);
		if (gSuccess==FALSE) return GEN_FAIL;   				
		
	if(gInfo.isWormInLoop==TRUE)
	{
		// This case will be implemented by Jay and sent to San Diego
//JF: Do these codes work?		
		// findLoopNodeWithHeaviestBranch(binImg, height, width); 		
		//resetExtractBackboneFromBinImg(); 
		resetCurrInfo() ;
		return GEN_FAIL;
		
	}
	
	else // no loop
	{
		pruneImage(binImg, height, width, &gInfo, -1, -1);
	
			
		//#ifdef DEBUG
		//	showImage(binImg, height, width, 1);
		//#endif	
				
			
		sampleSkel( binImg,  height, width,backboneY, backboneX, &gInfo.PsOI[gInfo.headPOI], &gInfo.PsOI[gInfo.tailPOI], numPixToRepBackbone);
		
		if( gIsPrevInfoValid )		
		{
			switchHeadAndTailIfNecessary();
		}	
		
		#ifdef DEBUG_SWITCH_HEAD_AND_TAIL
			showImageWithGray( binImg, height, width, 1, gInfo.BB.sample[0].y, gInfo.BB.sample[0].x, SAMPLE_POINTS_DISPLAY_COLOR );  
			Delay (DELAY_TIME_FOR_SHOWING_WHERE_TAIL_IS);
			//showImageWithGray( binImg, height, width, 1, gInfo.BB.sample[gInfo.BB.numSamples-1].y, gInfo.BB.sample[gInfo.BB.numSamples-1].x, SAMPLE_POINTS_DISPLAY_COLOR );  
			//Delay (.3);
			showImageWithGrayPix( binImg, height, width, 1, gInfo.BB.sample, gInfo.BB.numSamples, SAMPLE_POINTS_DISPLAY_COLOR );
			Delay (DELAY_TIME_FOR_DISPLAY_OF_FINAL_PRODUCT);
		#endif
	
		resetCurrInfo();	
		
		return SUCCESS;
	}
}


// ***************************************** THIN IMAGE *********************************** //
/* JF
int thinImg(char ** binImg, int height, int width)
{	   
	// returns wasAltered: whether or not binImg was altered

	char vicinity[NUM_DIRS+1];
	int row;
	int col;
	int wasImgAltered = 0;
	
	// 1st iteration
	for(row = 0+PAD; row<(height-PAD); row++)
	{
		for(col=0+PAD; col<(width-PAD); col++)
		{
			if(binImg[row][col])
			{
				makeVicinity(binImg, row, col, vicinity);
				if( thinCondition('a',vicinity) )
				{
					if( thinCondition('b',vicinity) )
					{
						if( thinCondition('c',vicinity) )
						{
							if( thinCondition('d',vicinity) ) 
							{
								binImg[row][col]=BACKGROUND;
								wasImgAltered = TRUE;
							}
						}
					}
				}
			}
		}
	}
	
	// 2nd iteration (only conditions c and d have been changed)	
	for(row = 0+PAD; row<(height-PAD); row++)
	{
		for(col=0+PAD; col<(width-PAD); col++)
		{
			if(binImg[row][col])
			{
				makeVicinity(binImg, row, col, vicinity);
				if( thinCondition('a',vicinity) )
				{
					if( thinCondition('b',vicinity) )
					{
						if( thinCondition('e',vicinity) )
						{
							if( thinCondition('f',vicinity) ) 
							{
								binImg[row][col]=BACKGROUND;
								wasImgAltered = TRUE;
							}
						}
					}
				}
			}
		}
	}
	return wasImgAltered;
}

JF */

/* JF
int thinCondition( char cond, char * P )
//      P8  P1	P2
//		    
//		P7	P0  P3     <-----  P[9] is a 3x3 neighborhood
//
//		P6	P5	P4

{
int nonZeroNeighs = 0;
int patterns01 = 0;
int i;
	switch(cond)
	{
		case 'a':
			for(i=1;i<=8;i++)
			{
				nonZeroNeighs += P[i];
			}
			if ( (2<=nonZeroNeighs)&(nonZeroNeighs<=6) )
				return TRUE;
			else
				return FALSE;
			break;
		case 'b':
			for(i=1;i<=7;i++)
			{
				if( (P[i]==0)&(P[i+1]==1) )
					patterns01 += 1;
			}
			if( (P[8]==0)&(P[1]==1) )
				patterns01 += 1;
			if(patterns01==1)
				return TRUE;
			else
				return FALSE;
			break;
		case 'c':
			if ( (P[1]*P[3]*P[5])==0 )
				return TRUE;
			else
				return FALSE;
			break;
		case 'd':
			if ( (P[3]*P[5]*P[7])==0 )
				return TRUE;
			else
				return FALSE;
			break;				
		case 'e':
			if ( (P[1]*P[3]*P[7])==0 )
				return TRUE;
			else
				return FALSE;
			break;
		case 'f':
			if ( (P[1]*P[5]*P[7])==0 )
				return TRUE;
			else
				return FALSE;
			break;
		default:
			return GEN_FAIL;
	}
	return SUCCESS;
}

*/
// ************************************************************************************************ //
// *************************** extractBackboneFromBinImg HELPER FUNCTIONS ************************* //
// ************************************************************************************************ //


// ************************ FIND POINTS OF INTEREST *********************************//
int findPointsOfInterest(
							// in
							char ** img,
							int height,
							int width,
							//out
							PointOfInterest * PsOI,
							int * numPsOI
						)
							
{
int i;
int rowIndex;
int colIndex;
int dirIndex;
int fanDirIndex;
int numNeighs;
int numImmedNeighs;
int poiIndex;
char vicinity[NUM_DIRS+1]; // one for all 8 directions +1 for the middle (index 0)
int cameFrom;
double totalDistance;

//JF
int iDebug = SUCCESS; 
//char sMessage[200];
//JF

*numPsOI = 0;



	for (rowIndex = PAD; rowIndex<(height-PAD); rowIndex++)
	{
		for(colIndex = PAD; colIndex<(width-PAD); colIndex++)
		{
			if(img[rowIndex][colIndex]==FOREGROUND)
			{
				#ifdef DEBUG_DIAGONALIZE
   					showImageWithGray(img,height,width,1,rowIndex,colIndex,150);
   				#endif

			

	   			makeVicinity(img, rowIndex, colIndex, vicinity);
	   			numNeighs = 0;
			
	   		
	   			if (isPoi(vicinity, &numNeighs)) // if this is a poi (PsOI have 1, 3, or 4 neighbors)
	   			{
				
					#ifdef THERE_MAY_BE_FANS
						if(vicinity[N_DIR])
						{
							if(vicinity[NE_DIR])
							{
								if(vicinity[E_DIR])
								{
									img[rowIndex-1][colIndex  ] = BACKGROUND;		
									img[rowIndex  ][colIndex+1] = BACKGROUND;		
								}
							}
						}
						if(vicinity[E_DIR])
						{
							if(vicinity[SE_DIR])
							{
								if(vicinity[S_DIR])
								{
									img[rowIndex  ][colIndex+1] = BACKGROUND;		
									img[rowIndex+1][colIndex  ] = BACKGROUND;											
								}
							}
						}
						if(vicinity[S_DIR])
						{
							if(vicinity[SW_DIR])
							{
								if(vicinity[W_DIR])
								{
									img[rowIndex+1][colIndex  ] = BACKGROUND;		
									img[rowIndex  ][colIndex-1] = BACKGROUND;											
								}
							}
						}

						if(vicinity[W_DIR])
						{
							if(vicinity[NW_DIR])
							{
								if(vicinity[N_DIR])
								{
									img[rowIndex  ][colIndex-1] = BACKGROUND;			
									img[rowIndex-1][colIndex  ] = BACKGROUND;											
								}
							}
						}
					
					
					#endif	   			


	   				if(*numPsOI == MAX_NUM_PSOI)
	   				{
	   					return GEN_FAIL;
	   				}
	   				*numPsOI = *numPsOI + 1;
	   				if (((numNeighs == 1)|(numNeighs == 3))|(numNeighs==4))
	   				{
	   					PsOI[*numPsOI-1].numNeighs = numNeighs;
	   					PsOI[*numPsOI-1].coord.x = colIndex;
	   					PsOI[*numPsOI-1].coord.y = rowIndex;
	   					PsOI[*numPsOI-1].isInLoop = 0;
						for(i=1;i<=NUM_DIRS;i++)	   					
	   					{
	   						PsOI[*numPsOI-1].which[i] = -1; 
	   						PsOI[*numPsOI-1].dist[i] = -1;
	   						PsOI[*numPsOI-1].rev[i] = -1;
	   					}
	   				}
	   				else if (numNeighs == 0)
	   				{
	   					img[rowIndex][colIndex] = BACKGROUND;
	   				}
	   				else
	   				{
	   					return GEN_FAIL;
	   				}
	   				if (numNeighs == 0)
	   				{
	   					img[rowIndex][colIndex] = BACKGROUND;
	   				}
	   			}
	   			else
	   			{
   					for (dirIndex=1; dirIndex<8; dirIndex+=2)    
   					{
   						if(vicinity[dirIndex]&vicinity[(int)fmod(dirIndex+2,8)])
   						{
							img[rowIndex][colIndex]=BACKGROUND;
						}
   					}
	   			}
	   		}
	   	}
	}
	#ifdef DEBUG
   		showImage(img,height,width,1);
   	#endif

   	for( poiIndex = 0; poiIndex < *numPsOI; poiIndex++ )
   	{
   	
   		makeVicinity(img, PsOI[poiIndex].coord.y, PsOI[poiIndex].coord.x, vicinity);
	
		for (dirIndex = 1; dirIndex < 8; dirIndex += 2)
	   	{

	   		if (vicinity[dirIndex])
	   		{

	   			gSuccess = exploreDir( img, &PsOI[poiIndex], dirIndex, numPsOI, PsOI, height, width ); 		
	   				if(gSuccess==FALSE) return GEN_FAIL;

	   		}
	   	}
	   	
		if( (!(vicinity[N_DIR]|vicinity[E_DIR]))&(vicinity[NE_DIR]) )
			gSuccess = exploreDir( img, &PsOI[poiIndex], NE_DIR, numPsOI, PsOI, height, width ); 			
			
		if( (!(vicinity[E_DIR]|vicinity[S_DIR]) )&(vicinity[SE_DIR]))
			gSuccess = exploreDir( img, &PsOI[poiIndex], SE_DIR, numPsOI, PsOI, height, width ); 			

		if( (!(vicinity[S_DIR]|vicinity[W_DIR]))&(vicinity[SW_DIR]) )
			gSuccess = exploreDir( img, &PsOI[poiIndex], SW_DIR, numPsOI, PsOI, height, width ); 
		
		if( (!(vicinity[W_DIR]|vicinity[N_DIR]))&(vicinity[NW_DIR]) )
			gSuccess = exploreDir( img, &PsOI[poiIndex], NW_DIR, numPsOI, PsOI, height, width ); 
			
		if(gSuccess==FALSE) 
		{
			iDebug = GEN_FAIL;
		}else iDebug = SUCCESS;

	}   

	return iDebug; 
}

// ************************ MAX DISTANCE ******************************* //
int maxDist(char ** img, int height, int width)
{
	int i;
	int poiHead;
	int poiTail;
	int numPsOI;
	double dists[MAX_NUM_PSOI/2+2];
	int head = -1;
	int tail = -1;
	char restrictedDirs[3];
	
	Map map[MAX_NUM_PSOI/2+2];
	
	numPsOI = gInfo.numPsOI;	
	restrictedDirs[0]=0;	
	restrictedDirs[1]=0;	
	restrictedDirs[2]=0;	
	
	for(poiHead=0;poiHead<numPsOI;poiHead++)
	{	
	  if(gInfo.PsOI[poiHead].numNeighs==END_POINT)
	  {
		gSuccess = distFromHereToEverywhere( poiHead, gInfo.PsOI, dists, restrictedDirs, map, &gInfo.isWormInLoop, img, height, width );
			if (gSuccess==FALSE) return 0;
		
		
		for	(poiTail=0;poiTail<numPsOI;poiTail++)
		{   
		  if((gInfo.PsOI[poiTail].numNeighs==END_POINT)&(poiTail!=poiHead))
	      {
			if(gInfo.LD<dists[poiTail])
			{
				gInfo.LD = dists[poiTail];
				gInfo.headPOI = poiHead;
				gInfo.tailPOI = poiTail;
				gInfo.mapLD.numInsts = map[poiTail].numInsts;
				for(i=0;i<map[poiTail].numInsts;i++)
				{
					gInfo.mapLD.inst[i]=map[poiTail].inst[i];
				}
			}
		  }
		}
	  }	
	}
	return SUCCESS;
}

// *********************************** PRUNE IMAGE ************************************ //
int	pruneImage(char ** img, int height, int width, AnalyzedBackboneInfo * info, int loopPOI, int dirToIgnore)
{
	int mapIndex;
	int dirIndex;
	int cameFrom;
	int currPOI;
	int prevPOI;
	int nextPOI;
	int delX;
	int delY;
	int tempX;
	int tempY;
	int dontDeleteY;
	int dontDeleteX;
	double dummyVariable;
	char vicinity [NUM_DIRS+1];
	currPOI = info->headPOI;
	prevPOI = -1;

	
	
	delX = 0;
	delY = 0;
	for(mapIndex=0;mapIndex<info->mapLD.numInsts;mapIndex++)
	{
		for(dirIndex=1;dirIndex<=NUM_DIRS;dirIndex++)
		{	
		// HEY JAY!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
		// somewhere around here see if you can restrict the delete all function by 
		// checking which directions the point of interest has something in (.which[])
		// see paper **10 on desk
			nextPOI = info->PsOI[currPOI].which[info->mapLD.inst[mapIndex]];
			if( (!((loopPOI==currPOI)&(dirToIgnore==dirIndex))) & ((info->PsOI[currPOI].which[dirIndex])>=0) )
			{
				if((info->PsOI[currPOI].which[dirIndex]!=nextPOI)&(info->PsOI[currPOI].which[dirIndex]!=prevPOI))
				{
					tempX = info->PsOI[currPOI].coord.x;
					tempY = info->PsOI[currPOI].coord.y;
					dontDeleteX = tempX;
					dontDeleteY = tempY;
					move( dirIndex, &cameFrom, &tempY, &tempX );
					if(img[tempY][tempX])
					{
						//if (!( (tempY==(PsOI[info->PsOI[currPOI].which[dirIndex]].coord.y)
						//   & (tempY==(PsOI[info->PsOI[currPOI].which[dirIndex]].coord.y) ))
						delY=tempY;
						delX=tempX;
						
						img[delY][delX] = BACKGROUND;				
						makeVicinity(img, delY, delX, vicinity); 
						moveForward( &dummyVariable, &cameFrom, &delY, &delX, vicinity );
						deleteAllTouching(img,delY,delX,dontDeleteY,dontDeleteX);
					}
				}
			}
		}
		prevPOI = currPOI;
		currPOI = nextPOI;
		#ifdef DEBUG_PRUNE
			showImageWithGray(img, height, width, 1, delY, delX, 100);
		#endif
	}
	return SUCCESS;
}
// ******************************* DELETE ALL TOUCHING ********************************* //
int deleteAllTouching(char ** img, int y, int x, int dontDeleteY, int dontDeleteX)
{
	int i;
	int xRec;
	int yRec;
	int yBackup;
	int xBackup;
	int dummyVariable;
	char vicinity[NUM_DIRS+1];
	makeVicinity(img, y, x, vicinity);
	img[y][x]=BACKGROUND;
	for(i=1;i<=NUM_DIRS;i++)
	{
		if(vicinity[i])
		{
			yRec = y;
			xRec = x;
			move( i, &dummyVariable, &yRec, &xRec ); 
			if(!((dontDeleteY==yRec)&(dontDeleteX==xRec)))
			{
				deleteAllTouching(img,yRec,xRec,dontDeleteY,dontDeleteX);
			}
		} 
	}
	return SUCCESS;
}

// ******************************* DISTANCE FROM HERE TO EVERYWHERE *********************************//
int distFromHereToEverywhere( int startPOI, PointOfInterest * PsOI, double * dists, char * restrictedDir, Map * map, int * isWormInLoop, char ** img, int height, int width )
//restricted dir should be 0 if there is no direction restriction on the starting move

{

int instIndex;

int cameFromPOI;
char dirTraveledStack[(int) (MAX_NUM_PSOI/2+2)];  // a history of which ways we've hopped indexed by pad number
int visitedNodes[(int) (MAX_NUM_PSOI/2+3)]; // a list of the nodes we've visited so far   
char dirToTry[(int) (MAX_NUM_PSOI/2+2)]; // a record of which direction we've already tried from each pad indexed by pad number
double currDist = 0; // a running sum of the distance from starting point to the current pad
int currPOI; // tells which POI the current pad corresponds to
int currNode=0; // tells how many pads we've hopped on
int allEndPointsHaveBeenVisited = FALSE;
int i;
int haveBeenHereBefore = FALSE;
currPOI=startPOI;
visitedNodes[0] = currPOI; 
dirToTry[0]=1;

	while(!allEndPointsHaveBeenVisited)
	{
		#ifdef DEBUG_DISTS
			showImageWithGray(img, height, width, 1, PsOI[currPOI].coord.y, PsOI[currPOI].coord.x, 100);    
		#endif
		haveBeenHereBefore = FALSE;
		//check to see if we've been here before
		if (!(dirToTry[currNode]>NUM_DIRS))
		{
			if( cameFromPOI!=PsOI[currPOI].which[dirToTry[currNode]] )
			{
				for (i=0;i<(currNode);i++)
				{
					if (visitedNodes[i]==currPOI)
					{
						PsOI[currPOI].isInLoop=TRUE;
						*isWormInLoop=TRUE;
						haveBeenHereBefore=TRUE;
					}
				}
			}
		}
		if(haveBeenHereBefore)
		{
			// pop
			currNode -= 1;
			currPOI = visitedNodes[currNode];
			currDist = currDist - PsOI[currPOI].which[dirToTry[currNode]-1];
			for(instIndex=0;instIndex<currNode;instIndex++)
			{
				map[currPOI].inst[instIndex] = dirTraveledStack[instIndex];
			}
			map[currPOI].numInsts=currNode;
			dists[currPOI] = currDist;
			cameFromPOI = visitedNodes[currNode-1];    
		}
		
		else if(currNode==0) // we are at the starting node
		{
			if(dirToTry[currNode]>NUM_DIRS)
			{
				allEndPointsHaveBeenVisited = TRUE;
			}
			else if( (PsOI[startPOI].which[dirToTry[currNode]]>NOTHING_IN_THIS_DIR) 
					& (!((dirToTry[currNode]==restrictedDir[0])|(dirToTry[currNode]==restrictedDir[1])|(dirToTry[currNode]==restrictedDir[2]))) 
					)// there's something in the current direction and the current direction is not restricted and we didn't come from it
			{
				dirTraveledStack[currNode] = dirToTry[currNode]; // push new direction on map 
				visitedNodes[currNode+1]=PsOI[startPOI].which[dirToTry[currNode]]; // push a POI
				currDist += PsOI[startPOI].dist[dirToTry[currNode]]; // update distance
				cameFromPOI=currPOI;
				dirToTry[currNode] += 1; // update the next direction to try from this node when we come back to it later
				currPOI = visitedNodes[currNode+1]; 
				currNode += 1;
				dirToTry[currNode]=1;
			}
			else if( (PsOI[startPOI].which[dirToTry[currNode]]==NOTHING_IN_THIS_DIR) | ((dirToTry[currNode]==restrictedDir[0])|(dirToTry[currNode]==restrictedDir[1])|(dirToTry[currNode]==restrictedDir[2])) ) 
				// nothing in this direction or the direction is restricted
			{
				dirToTry[currNode] += 1;
			}
			else
				return GEN_FAIL;
		}
		else if( (PsOI[currPOI].numNeighs==END_POINT) | (dirToTry[currNode]>NUM_DIRS) ) // we have tried all directions from this point or we're at an endpoint.  Either way pop one off.
		{
			if (PsOI[currPOI].numNeighs==END_POINT) // if at endpoint record current information
			{
				for(instIndex=0;instIndex<currNode;instIndex++)
				{
					map[currPOI].inst[instIndex] = dirTraveledStack[instIndex];
				}
				map[currPOI].numInsts=currNode;
				dists[currPOI] = currDist;
			}
			
			// pop stuff and take a step back
			currNode -= 1;
			currPOI = visitedNodes[currNode];
			currDist = currDist - PsOI[currPOI].dist[dirToTry[currNode]-1];
			if(currNode!=0) cameFromPOI = visitedNodes[currNode-1];
			
		}
		else if( (PsOI[currPOI].which[dirToTry[currNode]]>NOTHING_IN_THIS_DIR) 
			& (cameFromPOI!=PsOI[currPOI].which[dirToTry[currNode]])) // there's something in the current direction and the current direction is not restricted and we didn't come from it
		{
			cameFromPOI=currPOI;
			dirTraveledStack[currNode] = dirToTry[currNode]; // push new direction on map 
			visitedNodes[currNode+1]=PsOI[currPOI].which[dirToTry[currNode]]; // push a POI
			currDist += PsOI[currPOI].dist[dirToTry[currNode]]; // update distance
			dirToTry[currNode] += 1; // update the next direction to try from this node when we come back to it later
			currPOI = visitedNodes[currNode+1]; 
			currNode += 1;
			dirToTry[currNode]=1;
		}
		else
		{
			dirToTry[currNode]++;
		}
	}
	return SUCCESS;
}






// ***************************** IS POI *********************************//  
int isPoi(char * vicinity, int  * numNeighsToUpdate)
{
int dirIndex;
int numNeighs = 0;

	for ( dirIndex = 1; dirIndex<=4; dirIndex++ )
	{
		numNeighs += vicinity[dirIndex*2-1];	
	}
	if( !(vicinity[N_DIR]|vicinity[E_DIR]) )
		numNeighs += vicinity[NE_DIR];			
	if( !(vicinity[E_DIR]|vicinity[S_DIR]) )
		numNeighs += vicinity[SE_DIR];			
	if( !(vicinity[S_DIR]|vicinity[W_DIR]) )
		numNeighs += vicinity[SW_DIR];			
	if( !(vicinity[W_DIR]|vicinity[N_DIR]) )
		numNeighs += vicinity[NW_DIR];			
	
	#ifdef DEBUG
		if (numNeighs>4 | numNeighs<0)// <-- this should really say	if (numNeighs>4 | numNeighs<1), but a lone pixel occasionally slips through
		{
			return GEN_FAIL;
		}
	#endif
	
	*numNeighsToUpdate = numNeighs;
	
	if (numNeighs != 2) // if this is a poi (PsOI have 1, 3, or 4 neighbors)
	{
		if (((numNeighs == 1)|(numNeighs == 3))|(numNeighs==4))
		{
			return TRUE;
		}
		else
		{
			return FALSE;
		}
	}
	else
	{
		return FALSE;
	}
	
}


// ***************************** MOVE *********************************//
int move( int dirToMove, int * cameFrom, int * y, int * x )
{
	switch(dirToMove)
	{
		case N_DIR:
			*y = *y + (-1);
			*x = *x + ( 0);
			*cameFrom = 5;
			break;
		case NE_DIR:
			*y = *y + (-1);
			*x = *x + ( 1);
			*cameFrom = 6;
			break;
		case E_DIR:  
			*y = *y + ( 0);
			*x = *x + ( 1);
			*cameFrom = 7;
			break;
		case SE_DIR: 
			*y = *y + ( 1);
			*x = *x + ( 1);
			*cameFrom = 8;
			break;
		case S_DIR:  
			*y = *y + ( 1);
			*x = *x + ( 0);
			*cameFrom = 1;
			break;
		case SW_DIR: 
			*y = *y + ( 1);
			*x = *x + (-1);
			*cameFrom = 2;
			break;
		case W_DIR:  
			*y = *y + ( 0);
			*x = *x + (-1);
			*cameFrom = 3;
			break;
		case NW_DIR: 
			*y = *y + (-1);
			*x = *x + (-1);
			*cameFrom = 4;
			break;
		default:
			return GEN_FAIL;
	}
	return SUCCESS;

}



// ***************************** MOVE FORWARD *********************************//
moveForward( double * totalDist, int * cameFrom, int * y, int * x, char * vicinity )
{

int nVic, neVic, eVic, seVic, sVic, swVic, wVic, nwVic;
nVic =vicinity[1];
neVic=vicinity[2];
eVic =vicinity[3];
seVic=vicinity[4];
sVic =vicinity[5];
swVic=vicinity[6];
wVic =vicinity[7];
nwVic=vicinity[8];

if (*cameFrom==N_DIR) 
{
      if (FOREGROUND==sVic)
            move(S_DIR, cameFrom, y, x);
        else if ( swVic )
            move(SW_DIR, cameFrom, y, x);
        else if ( seVic )
            move(SE_DIR, cameFrom, y, x);
        else
			return GEN_FAIL;
}
else if (*cameFrom==E_DIR)      
{
      if (FOREGROUND==wVic)
            move(W_DIR, cameFrom, y, x);
        else if ( nwVic )
            move(NW_DIR, cameFrom, y, x);
        else if ( swVic )
            move(SW_DIR, cameFrom, y, x);
        else
			return GEN_FAIL;
}
else if (*cameFrom==S_DIR)    
{
      if (FOREGROUND==nVic)
            move(N_DIR, cameFrom, y, x);
        else if ( nwVic )
            move(NW_DIR, cameFrom, y, x);
        else if ( neVic )
            move(NE_DIR, cameFrom, y, x);
        else
			return GEN_FAIL;
}
else if (*cameFrom==W_DIR)      
{
      if (FOREGROUND==eVic)
            move(E_DIR, cameFrom, y, x);
        else if ( neVic )
            move(NE_DIR, cameFrom, y, x);
        else if ( seVic )
            move(SE_DIR, cameFrom, y, x);
        else
			return GEN_FAIL;
}
else if (*cameFrom==NE_DIR)
{
      if (FOREGROUND==sVic)
            move(S_DIR, cameFrom, y, x);
        else if ( wVic )
            move(W_DIR, cameFrom, y, x);
        else if ( seVic )
            move(SE_DIR, cameFrom, y, x);
        else if ( swVic )
            move(SW_DIR, cameFrom, y, x);
        else if ( nwVic )
            move(NW_DIR, cameFrom, y, x);
        else
			return GEN_FAIL;
}			
else if (*cameFrom==SE_DIR)    
{
      if (FOREGROUND==nVic)
            move(N_DIR, cameFrom, y, x);
        else if ( wVic )
            move(W_DIR, cameFrom, y, x);
        else if ( nwVic )
            move(NW_DIR, cameFrom, y, x);
        else if ( swVic )
            move(SW_DIR, cameFrom, y, x);
        else if ( neVic )
            move(NE_DIR, cameFrom, y, x);
        else
 			return GEN_FAIL;
} 			
else if (*cameFrom==NW_DIR)
{
      if (FOREGROUND==sVic)
            move(S_DIR, cameFrom, y, x);
        else if ( eVic )
            move(E_DIR, cameFrom, y, x);
        else if ( seVic )
            move(SE_DIR, cameFrom, y, x);
        else if ( swVic )
            move(SW_DIR, cameFrom, y, x);
        else if ( neVic )
            move(NE_DIR, cameFrom, y, x);
        else
			return GEN_FAIL;
}
else if (*cameFrom==SW_DIR)    
{
      if (FOREGROUND==nVic)
            move(N_DIR, cameFrom, y, x);
        else if ( eVic )
            move(E_DIR, cameFrom, y, x);
        else if ( neVic )
            move(NE_DIR, cameFrom, y, x);
        else if ( seVic )
            move(SE_DIR, cameFrom, y, x);
        else if ( nwVic )
            move(NW_DIR, cameFrom, y, x);
        else
 			return GEN_FAIL;
}
else if (*cameFrom==0)    
{
      if (FOREGROUND==nVic)
            move(N_DIR, cameFrom, y, x);
        else if ( eVic )
            move(E_DIR, cameFrom, y, x);
        else if ( sVic )
            move(S_DIR, cameFrom, y, x);
        else if ( wVic )
            move(W_DIR, cameFrom, y, x);
        else if ( neVic )
            move(NE_DIR, cameFrom, y, x);
        else if ( seVic )
            move(SE_DIR, cameFrom, y, x);
        else if ( nwVic )
            move(NW_DIR, cameFrom, y, x);
        else if ( swVic )
            move(SW_DIR, cameFrom, y, x);            
        else
			return GEN_FAIL;
      
}
else
	return GEN_FAIL;

return SUCCESS;
}	


// ***************************** HOW FAR *********************************//
double howFar( int dir )
{
	switch(dir)
	{
		case N_DIR:
			return STRAIGHT_DIST;
			break;
		case NE_DIR:
			return DIAG_DIST;
			break;
		case E_DIR:  
			return STRAIGHT_DIST;
			break;
		case SE_DIR: 
			return DIAG_DIST;
			break;
		case S_DIR:  
			return STRAIGHT_DIST;
			break;
		case SW_DIR: 
			return DIAG_DIST;
			break;
		case W_DIR:  
			return STRAIGHT_DIST;
			break;
		case NW_DIR: 
			return DIAG_DIST;
			break;
		default:
			return GEN_FAIL;
	}
	return SUCCESS;
}


// ***************************** EXPLORE DIRECTION ********************************* //
int exploreDir( char ** img, PointOfInterest * startPoi, int firstMove, int * numPsOI, PointOfInterest * PsOI, int height, int width)
{
	char vicinity[NUM_DIRS+1];

	int poiIndex = 0;
	double totalDist = 0;
	int cameFrom = 0;
	int atEnd = 0;
	int foundPoi = 0;
	int yToMove = -1000; // nonsense default value to help catch error
	int xToMove = -1000; //
	int y;
	int x;                
	int numNeighs;
	
	y = startPoi->coord.y;
	x = startPoi->coord.x;

	#ifdef DEBUG_EXPLORE_DIR
		showImageWithGray(img, height, width, 1, y, x, 100);   
	#endif
	
	move( firstMove, &cameFrom, &y, &x );
	totalDist = totalDist + howFar( cameFrom );
	
	makeVicinity( img, y, x, vicinity );
	atEnd = isPoi(vicinity, &numNeighs);

	#ifdef DEBUG_EXPLORE_DIR
		showImageWithGray(img, height, width, 1, y, x, 100);   
	#endif
	
	while (!atEnd)
	{
		moveForward( &totalDist, &cameFrom, &y, &x, vicinity );
		totalDist = totalDist + howFar( cameFrom ); 
		makeVicinity( img, y, x, vicinity );
		atEnd = isPoi(vicinity, &numNeighs);
		#ifdef DEBUG_EXPLORE_DIR
			showImageWithGray(img, height, width, 1, y, x, 100);   
		#endif
	}
	
	startPoi->dist[firstMove] = totalDist;
	startPoi->rev[firstMove] = cameFrom;
	
	while( !foundPoi )
	{
		if(poiIndex==MAX_NUM_PSOI) // this is to tackle the case where thinning and
								   // other methods don't result in a reasonable 
								   // skeleton (goa_07_27_010.dat imageNode#1282
		{
			return GEN_FAIL;
		}
	
		if( PsOI[poiIndex].coord.x == x )
		{
			if( PsOI[poiIndex].coord.y == y )
			{
				startPoi->which[firstMove] = poiIndex;
				foundPoi = TRUE;
			}
		}
		poiIndex++;
	}
	
	
	
	return SUCCESS;
}

// ***************************** MAKE VICINITY *********************************//
int makeVicinity(char ** img, int row, int col, char * vicinity)
{
	vicinity[0] = img[row]  [col]  ;
	vicinity[1] = img[row-1][col]  ;
	vicinity[2] = img[row-1][col+1];
	vicinity[3] = img[row]  [col+1];
	vicinity[4] = img[row+1][col+1];
	vicinity[5] = img[row+1][col]  ;
	vicinity[6] = img[row+1][col-1];
	vicinity[7] = img[row]  [col-1];
	vicinity[8] = img[row-1][col-1];
	return SUCCESS;
}


// ***************************** PAD IMAGE *********************************//
/*JF
int padImg(char ** img, int height, int width, int padWidth)
{
	int rowIndex;
	int colIndex;
	
	// shift image by padWidth to the right and padWidth down
	for (rowIndex = height - 1; rowIndex >= 0; rowIndex--)
	{
		for (colIndex = width - 1; colIndex >= 0; colIndex--)
		{
			img[rowIndex+padWidth][colIndex+padWidth] = img[rowIndex][colIndex];
		}
	}
	
	// next 4 loops: pad image (set edges to bg value for padWidth)
	
	// pad top
	for (rowIndex = 0; rowIndex<padWidth; rowIndex++)
	{
		for (colIndex = 0; colIndex<(width+2*padWidth); colIndex++)
		{
			img[rowIndex][colIndex] = BACKGROUND;
		}
	}
	
	// pad bottom
	for (rowIndex = height+2*padWidth-1; rowIndex>(height+padWidth-1); rowIndex--)
	{
		for (colIndex = 0; colIndex<(width+2*padWidth); colIndex++)
		{
			img[rowIndex][colIndex] = BACKGROUND;
		}
	}
	
	// pad left
	for (colIndex = 0; colIndex<padWidth; colIndex++)
	{
		for (rowIndex = 0; rowIndex<(height+2*padWidth); rowIndex++)
		{
			img[rowIndex][colIndex] = BACKGROUND;
		}
	}
	
	// pad right
	for (colIndex = width+2*padWidth-1; colIndex>(width+padWidth-1); colIndex--)
	{
		for (rowIndex = 0; rowIndex<(height+2*padWidth); rowIndex++)
		{
			img[rowIndex][colIndex] = BACKGROUND;
		}
	}
	
	return SUCCESS;
}
*/
// **************************** FIND LOOP NODE WITH HEAVIEST BRANCH ******************* //
int findLoopNodeWithHeaviestBranch(char ** img, int height, int width)
{
	char restrictedDirs[3];
	int * dummyVariable;
	int poi;
	int dir;
	int endPoi;
	int mapIndex;
	int instIndex;
	int mapIndex2;
	int restrictIndex;
	int loopPOI;
	double weightOfTail;
	int error;
	
	Map map[(int)MAX_NUM_PSOI/2+2]; // map to all poi's sticking out from loop (the tail)
	double dists[(int)MAX_NUM_PSOI/2+2]; // the distances to each poi
	double greatestWeight;
	int poiGreatestWeight;

	
	
	
	
	for (poi=0;poi<gInfo.numPsOI;poi++)
	{
		
		if(gInfo.PsOI[poi].isInLoop)
		{
			// set all numInsts to zero (all that are pertinent)
			for(endPoi=0;endPoi<gInfo.numPsOI;endPoi++)
			{
				map[endPoi].numInsts = 0;
			}

			// do the following for every POI that's in the loop
			weightOfTail = 0;
			restrictIndex = 0;
			restrictedDirs[0]=0;
			restrictedDirs[1]=0;
			restrictedDirs[2]=0;
			for(dir=1;dir<=NUM_DIRS;dir++)
			{
				if(gInfo.PsOI[poi].which[dir]>=0)
				{
					if(gInfo.PsOI[gInfo.PsOI[poi].which[dir]].isInLoop)
					{   
						#ifdef DEBUG
							if (restrictIndex>2)
							{
								error = TRUE; // There are more than three directions that lead to loops, there could
											  // be double looping going on, or something else weird.
							}
						#endif
						
						restrictedDirs[restrictIndex]=dir;
						restrictIndex++;
					}
				}
			}
			
			distFromHereToEverywhere( poi, gInfo.PsOI, dists, restrictedDirs, map, dummyVariable, img, height, width );				
			
			weightOfTail = 0;                       
			for(mapIndex=0;mapIndex<gInfo.numPsOI;mapIndex++)
			{
				if(map[mapIndex].numInsts>0)
				{
					weightOfTail += dists[mapIndex];   
				}
			}
					
			if(weightOfTail>gInfo.tail.greatestWeight)
			{
				gInfo.intersectPOI = poi;
				gInfo.tail.greatestWeight = weightOfTail;
				for(mapIndex=0;mapIndex<gInfo.numPsOI;mapIndex++)   
				{
					gInfo.tail.dists[mapIndex] = dists[mapIndex];
					if(gInfo.PsOI[mapIndex].numNeighs==1)
					{
						if(map[mapIndex].numInsts>0)
						{	
							for(instIndex=0; instIndex<map[mapIndex].numInsts; instIndex++)
							{
								gInfo.tail.map[mapIndex].inst[instIndex] = map[mapIndex].inst[instIndex];
							}
							gInfo.tail.map[mapIndex].numInsts = map[mapIndex].numInsts;
						}
						else
						{
							gInfo.tail.map[mapIndex].numInsts = 0;
						}
					}
				}
			}
		}
	}
	
	return SUCCESS	;
}



// *************************************** SAMPLE SKELETON *********************************************** //
int sampleSkel( char ** img, int height, int width, double * backboneY, double * backboneX, PointOfInterest * headPOI, PointOfInterest * tailPOI, int numPixToRepBackbone)
{
	#ifdef DEBUG_SAMPLE
		int BBintX[MAX_PIXELS_IN_WORM];
		int BBintY[MAX_PIXELS_IN_WORM];
	#endif
	
	double totalDist;
	double distBtwSamples;
	double cumDist;
	int i;
	int numSamples;
	int distIndex;
	
	int currX;
	int currY;
	int cameFrom = 0;
	int reachedHead = 0;
	int timesVisitedIntersectNode = 0;
	double * dummyVariable;
	char vicinity[NUM_DIRS + 1];
	int dir;
				  
	int headX;
	int headY;
	headX = gInfo.PsOI[gInfo.headPOI].coord.x;
	headY = gInfo.PsOI[gInfo.headPOI].coord.y;
	
	gInfo.BB.numSamples = numPixToRepBackbone;
	
	gInfo.BB.numPix = 0;
	
	
	dummyVariable = 0; // ensure no wild pointers
	
	currX = gInfo.PsOI[gInfo.tailPOI].coord.x;
	currY = gInfo.PsOI[gInfo.tailPOI].coord.y;
	
	
	gInfo.BB.coord[gInfo.BB.numPix].x = currX;
	gInfo.BB.coord[gInfo.BB.numPix].y = currY;
	gInfo.BB.sample[0].x = currX;
	gInfo.BB.sample[0].y = currY;

	gInfo.BB.numPix ++;  
	
	
	
	while (!reachedHead)
	{
	
		#ifdef DEBUG_TRAVERSE
			showImageWithGray( img, height, width, 1, currY, currX, 100 );
		#endif
	
		makeVicinity(img, currY, currX, vicinity); 
		gSuccess = moveForward( dummyVariable, &cameFrom, &currY, &currX, vicinity );
		if(!gSuccess)
		{
			for(dir=1;dir<=NUM_DIRS;dir++)
			{
				if(vicinity[dir]&(cameFrom!=dir))
				{
					move(dir,&cameFrom,&currY,&currX);
					break;
				}
			}
		}
		
		gInfo.BB.coord[gInfo.BB.numPix].x = currX;
		gInfo.BB.coord[gInfo.BB.numPix].y = currY;
		gInfo.BB.dists[gInfo.BB.numPix-1] = howFar(cameFrom);
		gInfo.BB.numPix ++;  
		
		if( (currX == headX)&(currY == headY) )
		{
			if(gInfo.isWormInLoop)
			{
				timesVisitedIntersectNode++;
				if( timesVisitedIntersectNode == 2 )
				{
					reachedHead = TRUE;         	
				}
			}
			else
			{
				reachedHead = TRUE;         				
			}
		}
	
	}

	numSamples = 0;
	
	if(numPixToRepBackbone == 0)
		return GEN_FAIL;
	cumDist = 0;
	totalDist = 0;	
	backboneY[0] = gInfo.BB.coord[0].y;
	backboneX[0] = gInfo.BB.coord[0].x;
	#ifdef DEBUG_SAMPLE
		BBintY[0]= gInfo.BB.coord[0].y;
		BBintX[0]= gInfo.BB.coord[0].x;
	#endif
		
		
	numSamples++;	
	for(distIndex=0;distIndex<(gInfo.BB.numPix-1);distIndex++)
	{
		totalDist += gInfo.BB.dists[distIndex];
	}
	distBtwSamples = (totalDist)/((double)(numPixToRepBackbone-1));
	for(i=0;i<(gInfo.BB.numPix-1);i++)
	{
		
		cumDist += gInfo.BB.dists[i];
		if(cumDist>distBtwSamples)
		{
			cumDist = cumDist - distBtwSamples;//0 LOOK HERE AND DECIDE IF 0 OR cumDist - distBtwSamples
			backboneY[numSamples] = gInfo.BB.coord[i+1].y;
			backboneX[numSamples] = gInfo.BB.coord[i+1].x;
			gInfo.BB.sample[numSamples].y = gInfo.BB.coord[i+1].y;
			gInfo.BB.sample[numSamples].x = gInfo.BB.coord[i+1].x;
			#ifdef DEBUG_SAMPLE
				BBintY[numSamples]= gInfo.BB.coord[i+1].y;
				BBintX[numSamples]= gInfo.BB.coord[i+1].x;
			#endif
			numSamples++;
			#ifdef DEBUG_SAMPLE
				showImageWithGray( img, height, width, 1, BBintY[numSamples-1], BBintX[numSamples-1], 100 );  
			#endif
			
			if (numSamples == numPixToRepBackbone-1)
			{
				break;
			}
		}
	}
	backboneY[numPixToRepBackbone-1] = gInfo.BB.coord[gInfo.BB.numPix-1].y;
	backboneX[numPixToRepBackbone-1] = gInfo.BB.coord[gInfo.BB.numPix-1].x;
	gInfo.BB.sample[numPixToRepBackbone-1].y = gInfo.BB.coord[gInfo.BB.numPix-1].y;
	gInfo.BB.sample[numPixToRepBackbone-1].x = gInfo.BB.coord[gInfo.BB.numPix-1].x;
	#ifdef DEBUG_SAMPLE
		BBintY[numPixToRepBackbone-1]= gInfo.BB.coord[gInfo.BB.numPix-1].y;
		BBintX[numPixToRepBackbone-1]= gInfo.BB.coord[gInfo.BB.numPix-1].x;
	#endif
	#ifdef DEBUG_SAMPLE
		showImageWithGray( img, height, width, 1, BBintY[numPixToRepBackbone-1], BBintX[numPixToRepBackbone-1], 100 );  
	#endif


	return SUCCESS;
}

// ***************************** SWITCH HEAD AND TAIL IF NECESSARY ***************************************//
int switchHeadAndTailIfNecessary(void)
{
	double meanFwd;
	double meanRev;
	double varianceFwd;
	double varianceRev; 
	double distFwd[MAX_PIXELS_TO_REP_BACKBONE]; 
	double distRev[MAX_PIXELS_TO_REP_BACKBONE];
	iPoint  pixBuffer[MAX_PIXELS_IN_WORM]; 
	dPoint sampBuffer[MAX_PIXELS_TO_REP_BACKBONE];
	int i;
	int numSamples = gPrevInfo.BB.numSamples;
	
	if(gInfo.BB.numSamples != gPrevInfo.BB.numSamples)
	{
		return GEN_FAIL;
	}
	
	numSamples = gPrevInfo.BB.numSamples;            
	
	meanFwd = 0;
	meanRev = 0;
	varianceFwd = 0;
	varianceRev = 0;

	
	for(i=0;i<numSamples;i++)
	{
		distFwd[i] =   sqrt(  (   (gInfo.BB.sample[i].x - gPrevInfo.BB.sample[i].x)
						    	* (gInfo.BB.sample[i].x - gPrevInfo.BB.sample[i].x) )
							+ (   (gInfo.BB.sample[i].x - gPrevInfo.BB.sample[i].x)
						    	* (gInfo.BB.sample[i].x - gPrevInfo.BB.sample[i].x) )
						   );
		meanFwd+=distFwd[i];
		
		distRev[i] =   sqrt(  (   (gInfo.BB.sample[i].x - gPrevInfo.BB.sample[numSamples - i].x)
						    	* (gInfo.BB.sample[i].x - gPrevInfo.BB.sample[numSamples - i].x) )
							+ (   (gInfo.BB.sample[i].x - gPrevInfo.BB.sample[numSamples - i].x)
						    	* (gInfo.BB.sample[i].x - gPrevInfo.BB.sample[numSamples - i].x) )
						   );
		meanRev+=distRev[i];
	}
	meanFwd = meanFwd/numSamples;
	meanRev = meanRev/numSamples;
	
	for(i=0;i<numSamples;i++)
	{
		varianceFwd += (meanFwd - distFwd[i]) * (meanFwd - distFwd[i]);
		varianceRev += (meanRev - distRev[i]) * (meanRev - distRev[i]);
	}
	
	if( varianceRev < varianceFwd ) // if true reverse the direction of the worm
	{
		 // continuous backbone
		 for(i=0;i<gInfo.BB.numPix;i++)
		 {
		 	pixBuffer[i].y = gInfo.BB.coord[gInfo.BB.numPix-i].y;
		 	pixBuffer[i].x = gInfo.BB.coord[gInfo.BB.numPix-i].x;
		 }
		 for(i=0;i<gInfo.BB.numPix;i++)
		 {
		 	gInfo.BB.coord[i].y = pixBuffer[i].y;
		 	gInfo.BB.coord[i].x = pixBuffer[i].x;
		 }

		 // sampled backbone
		 for(i=0;i<numSamples;i++)
		 {
		 	sampBuffer[i].y = gInfo.BB.sample[numSamples-i].y;
		 	sampBuffer[i].x = gInfo.BB.sample[numSamples-i].x;
		 }
		 for(i=0;i<numSamples;i++)
		 {
		 	gInfo.BB.sample[i].y = sampBuffer[i].y;
		 	gInfo.BB.sample[i].x = sampBuffer[i].x;
		 }
		 
	
	}



	return SUCCESS;
}



// ************************************************************************************************ //
// ************************************	INSTANTIATION FUNCTIONS *********************************** //
// ************************************************************************************************ //







// ************************************************************************************************ //
// ************************************	DESTRUCTION FUNCTIONS ************************************* //
// ************************************************************************************************ //

/*JF
int passCurrInfoToPrev(void)
{
	gIsPrevInfoValid = TRUE;
	
	// using memcopy, if any pointers are ever used in the data structure
	// REMEMBER TO REWORK THE MEMCOPY PORTION OF THE CODE
	
	memcpy(&gPrevInfo,&gInfo,sizeof(AnalyzedBackboneInfo));
	return SUCCESS;
}
*/

// **************** RESET INFO *****************//
int resetCurrInfo( void )
{	
	int i;

	// tail
	for(i=0;i<gInfo.numPsOI;i++)
	{
		gInfo.tail.map[i].numInsts = 0;
		gInfo.PsOI[i].isInLoop = 0;
	}
	gInfo.tail.greatestWeight = 0;
	gInfo.tail.poiGreatestWeight = -1;

	
	// Backbone
	gInfo.BB.numPix = 0;
	gInfo.BB.numSamples = 0;

	// main
	gInfo.numPsOI = -1;          
	gInfo.isWormInLoop = 0;
	gInfo.headPOI=-1;
	gInfo.tailPOI = -1;
	gInfo.LD = -1;
	gInfo.mapLD.numInsts = 0;
	
	return SUCCESS;

}



// **************** destroy function ******************* //  
/*
int resetExtractBackboneFromBinImg()
{
	gIsPrevInfoValid = 0;
	return 1;
}

*/

// ************************************************************************************************ //
// ****************************************** DEBUG FUNCTIONS ************************************* //
// ************************************************************************************************ //

#ifdef DEBUG

void char2dToChar1d( char * img1d, char ** img2d, int height, int width )
{
	int rowIndex;
	int colIndex;
	
	for (rowIndex = 0; rowIndex<height; rowIndex++)
	{
		for (colIndex = 0; colIndex<width; colIndex++)
		{	
			img1d[rowIndex*width+colIndex] = 255*img2d[rowIndex][colIndex];
		}
	}
}
#endif

#ifdef DEBUG 
 void showImage( char ** char2d, int height, int width, int windowNumber )
{
	int rowIndex;
	char * char1d;
	Image * iImg;
	int windowHandle;
	int bitmapSmall;
	
	// mem allocation
	char1d = malloc( width*height*sizeof(char) );
	
	iImg = imaqCreateImage (IMAQ_IMAGE_U8, 0);

	// show image
	char2dToChar1d(char1d, char2d, height, width);
	imaqArrayToImage (iImg, char1d, width, height);
	imaqScale (iImg, iImg, 3, 3, IMAQ_SCALE_LARGER, IMAQ_NO_RECT);
	
	
//	imaqGetWindowHandle( &windowHandle );
//	imaqDisplayImage (iImg, windowHandle, TRUE); 
	
	// mem deallocation

	free( char1d );
	if (!gWindowExists[windowNumber])
	{
		rowIndex = IMAQ_WIND_RESIZABLE;
		rowIndex = IMAQ_WIND_TOPMOST;		
		rowIndex = IMAQ_WIND_TITLEBAR;
		gSuccess = imaqSetupWindow (windowNumber,IMAQ_WIND_TOPMOST);	
		imaqSetWindowSize (windowNumber, 100, 100);
		gWindowExists[windowNumber] = 1;
	}
	
	

	gSuccess = imaqDisplayImage (iImg, windowNumber, TRUE);
	//gSuccess = imaqShowWindow (windowNumber, TRUE);
	//ProcessSystemEvents();
	imaqDispose (iImg);
}  
#endif
#ifdef DEBUG

 void showImageWithGray( char ** char2d, int height, int width, int windowNumber, int rowGray, int colGray, char grayVal )
{

	int rowIndex;
	char * char1d;
	Image * iImg;
	int windowHandle;
	int bitmapSmall;
	char beforeItWasGray;
	int invert;
	
	// mem allocation
	char1d = malloc( width*height*sizeof(char) );
	
	iImg = imaqCreateImage (IMAQ_IMAGE_U8, 0);

	// show image
	
	
	
	
	for(invert=-7;invert<8;invert++)
	{
		if(((rowGray+invert)>=0)&((rowGray+invert)<height))
		{
		char2d[rowGray+invert][colGray] = !char2d[(rowGray+invert)][colGray];
		}
	}
	for(invert=-7;invert<0;invert++)
	{
		if(((colGray+invert)>=0)&((colGray+invert)<width))
		{
			char2d[rowGray][colGray+invert] = !char2d[rowGray][(colGray+invert)];
		}
	}
	for(invert=0;invert<8;invert++)
	{
		if(((colGray+invert)>=0)&((colGray+invert)<width))
		{
			char2d[rowGray][colGray+invert] = !char2d[rowGray][(colGray+invert)];
		}
	}
	
	
	char2dToChar1d(char1d, char2d, height, width);
	
	for(invert=-7;invert<8;invert++)
	{
		if(((rowGray+invert)>=0)&((rowGray+invert)<height))
		{
		char2d[rowGray+invert][colGray] = !char2d[(rowGray+invert)][colGray];
		}
	}
	for(invert=-7;invert<0;invert++)
	{
		if(((colGray+invert)>=0)&((colGray+invert)<width))
		{
			char2d[rowGray][colGray+invert] = !char2d[rowGray][(colGray+invert)];
		}
	}
	for(invert=0;invert<8;invert++)
	{
		if(((colGray+invert)>=0)&((colGray+invert)<width))
		{
			char2d[rowGray][colGray+invert] = !char2d[rowGray][(colGray+invert)];
		}
	}

	imaqArrayToImage (iImg, char1d, width, height);
	imaqScale (iImg, iImg, 3, 3, IMAQ_SCALE_LARGER, IMAQ_NO_RECT);
	
	
//	imaqGetWindowHandle( &windowHandle );
//	imaqDisplayImage (iImg, windowHandle, TRUE); 
	
	// mem deallocation

	free( char1d );
	if (!gWindowExists[windowNumber])
	{
		rowIndex = IMAQ_WIND_RESIZABLE;
		rowIndex = IMAQ_WIND_TOPMOST;		
		rowIndex = IMAQ_WIND_TITLEBAR;
		gSuccess = imaqSetupWindow (windowNumber,IMAQ_WIND_TOPMOST);	
		imaqSetWindowSize (windowNumber, 100, 100);
		gWindowExists[windowNumber] = 1;
	}
	
	

	gSuccess = imaqDisplayImage (iImg, windowNumber, TRUE);
	//gSuccess = imaqShowWindow (windowNumber, TRUE);
	//ProcessSystemEvents();
	imaqDispose (iImg);
	#ifdef DEBUG_PROCESS_SYSTEM_EVENTS
		ProcessSystemEvents();
	#endif
}  


void showImageWithGrayPix( char ** char2d, int height, int width, int windowNumber, dPoint * pointGray, int numPix, char grayVal )      
{

	int rowIndex;
	char * char1d;
	Image * iImg;
	int windowHandle;
	int bitmapSmall;
	char beforeItWasGray;
	int invert;
	int i;
	
	// mem allocation
	char1d = malloc( width*height*sizeof(char) );
	
	iImg = imaqCreateImage (IMAQ_IMAGE_U8, 0);

	// show image
	
	for(i=0;i<numPix;i++)
	{
		char2d[(int)pointGray[i].y][(int)pointGray[i].x] = grayVal;
	}
	
	
	char2dToChar1d(char1d, char2d, height, width);
	

	imaqArrayToImage (iImg, char1d, width, height);
	imaqScale (iImg, iImg, 3, 3, IMAQ_SCALE_LARGER, IMAQ_NO_RECT);
	
	
//	imaqGetWindowHandle( &windowHandle );
//	imaqDisplayImage (iImg, windowHandle, TRUE); 
	
	// mem deallocation

	free( char1d );
	if (!gWindowExists[windowNumber])
	{
		rowIndex = IMAQ_WIND_RESIZABLE;
		rowIndex = IMAQ_WIND_TOPMOST;		
		rowIndex = IMAQ_WIND_TITLEBAR;
		gSuccess = imaqSetupWindow (windowNumber,IMAQ_WIND_TOPMOST);	
		imaqSetWindowSize (windowNumber, 500, 500);
		gWindowExists[windowNumber] = 1;
	}
	
	

	gSuccess = imaqDisplayImage (iImg, windowNumber, TRUE);
	//gSuccess = imaqShowWindow (windowNumber, TRUE);
	//ProcessSystemEvents();
	imaqDispose (iImg);
	#ifdef DEBUG_PROCESS_SYSTEM_EVENTS
		ProcessSystemEvents();
	#endif
}  




#endif  
