function [x, y, timeV] = preorient(directory)

%PREORIENT    Automatically orients (heads & tails) worm data
%   [x,y] = PREORIENT(directory) 
%      where 
%      - "x" and "y" are matrices of x and y coordinates, 
%         automatically oriented with the computer-identified
%         head-end in column 1.  
%      - "timeV" is a vector representing the time (in seconds, starting
%         with zero) that each row of x and y was digitized.  (For 
%         analog-source data  or other data without timestamps, timeV
%         is an empty matrix.)
%      - "directory" is the name of a directory containing a file 
%         called 'points' (generated by RECOGNIZER*).  ("directory"
%         will also typically contain a series of corresponding 
%         image files to be used by MANPROC.)
%
%      N.B.  This funtion assumes that an omega turn is identified by 
%      a range of missing frames at least FIVE frames long.  Worm head-end
%      is defined by a quorum of foraging behavior, head-end curls to 
%      center of body when entering an omega turn, and tail-end remains 
%      curled to body when exiting an omega turn.
% 
%   C. J. Cronin 6-03-03
%   Revised 07-23-04 CJC.
%   $Revision: 2.01 $  $Date: 2004/07/23 xx:xx:xx $
%

% r2.01: Set up to automatically handle input data with or without 
%   timestamp.  Changes '\' to filesep for portability to non-
%   Windows platforms.  Added 'timeV' to function output.
%   Dated 7-23-04 x:xxPM.
%
% r1.02: Corrects to accept any number of points distributed along
%   worm length.
%   Dated 11-11-03 3:43PM.
%
% r1.01: Initial release.
%   Dated 7-21-03 3:29PM.

% Precondition the function call's input
% % % global directory
% % % if isempty(directory)
% % %     disp('lkdjf;lksadj fl;akj;lksdjfl;kjlkf')
% % % end

if nargin ~= 1   % nothing specified in preorient call
    error('ERROR-- Syntax: PREORIENT(directory)');
end
% if nargin < 1   % nothing specified in preorient call
%     error('ERROR-- Syntax: PREORIENT(directory) or PREORIENT(directory, subsamplingrate)');
% end

% % %     if exist('directory') ~= 1  % use existing directory, if established
% % %         directory = input('Enter Directory name> ','s');    % else prompt
% % %     end

% if nargin < 2   % if no subsampling rate is specified
%     subsamplerate = 1;
% end

omega_range = 5;    % Minimum number of sequential invalid frames identifying
                    %   an alleged "omega" turn.


    
% load([directory '\points']);
% imagedir = dir([directory filesep 'file.*.jpg']);
% temp_image_name = imagedir(i).name;

% Find file(s) named points* in input directory:
pointsname = dir([directory filesep 'points*']);

% Take first points* file as our choice:
pointsname = pointsname(1).name;

% Load selected directory 
points = load([directory filesep pointsname]);

% Determine whether points contains timestamp data:
% Create presumptive time vector and parity check vector
timeV = points(1:2:end,1);      % (presumptive) TIMESTAMP vector
timeV_parity = points(2:2:end,1);      % parity-check TEMPORARY TIMESTAMP vector


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Check for Timestamp and populate x-y matrices:
%    Verify presumptive timestamps are pairwise identical AND never
%    decreasing over time:
if ( sum( abs( timeV - timeV_parity) ) == 0 ) & ( all(diff(timeV) >=0) )   
        
    % Populate x & y matrices, skipping timestamp in first column:
    x = points(1:2:end,2:end);      % x-data in odd rows of points - SKIP TIMESTAMP
    y = points(2:2:end,2:end);      % y-data in even rows of points - SKIP TIMESTAMP
    
else    % if either test above fails, then assume no timestamp
    fprintf(1,'\n\tNOTE: Data appears to be from an analog source: \n\t\t\tNo Timestamp data identifiable in ''points'' file\n');
    timeV = [];     % Reset timeV to a placeholder if no timestamp is present
    % Populate x & y matrices:
    x = points(1:2:end,1:end);      % x-data in odd rows of points 
    y = points(2:2:end,1:end);      % y-data in even rows of points 
    
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

clear timeV_parity 


nworms = size(x,1);     % number of frames
middlepoint = size(x,2)/2 + 0.5;    % Index of mid-point on worm.  Integer if
                                    % odd number of points, x.5 if even number
                                    % of points.  Use for checking head 

% Initialize valid-flag vector
valid = ~isnan(x(:,1));     % Initialize valid flags to 1's (i.e. 
                            %   "yes, valid" for coordinates with 
                            %   numeric data; set to 0's (i.e. 
                            %   "invalid" for coordinates without data.
                                
% Create vector of worm lengths
len = sum(sqrt(diff(x').^2 + diff(y').^2)); % Corrected...

% Find Most Common Length
lengths = [1:1:1000];   % Reference vector (wormlengths to 1000 pixels)
h = hist(len,lengths);  % Actual wormlengths into ref vector bins

indx = find( h == max(h) );     % Most common worm length(s)
indx = indx(1);             % The first (shortest), most common length
                            %   (to handle the rare case when there are
                            %   two or more most common lengths)-- (if
                            %   the first is not the correct choice, the
                            %   user will need to manually screen the 
                            %   data, or will need to modify this code.
                            
mcl = lengths(indx);        % MOST COMMON WORM LENGTH, in pixels

% Worm-length band-pass filter:
shortlength = 0.89*mcl;                         % 89% of most common length
longlength = 1.12*mcl;                          % 112% of most common length
valid = valid & ((len' >= shortlength) & (len' <= longlength));   


%-----------------------------------------------------------
% Orient each worm with repect to the worm before 
for i = 2:nworms
    if flip( x(i,:), y(i,:),   x(i-1,:), y(i-1,:))  % Compare current w/previous frame
        % Reverse i'th worm; i+1'th will flip based on i'th
        x(i,:) = x(i,end:-1:1);       % Reorder worm
        y(i,:) = y(i,end:-1:1);  
    end     % if flip(...
end     % for i = 2:nworms
    

%-----------------------------------------------------------
% Double check for flips where single worm length is 
%   grossly under MCL
for i = 3:nworms
    if (len(i-1) < 0.2*mcl) & (valid(i) & valid(i-2)) 
        % Compare current w/frame before ultra-short worm
        if flip( x(i,:), y(i,:),   x(i-2,:), y(i-2,:))  
            % This time toggle flip from i to end of data set
            x(i:end,:) = x(i:end,end:-1:1);       % Reorder worm 
            y(i:end,:) = y(i:end,end:-1:1);  
        end     % if flip(...
    end     % if (len(i-1)...
end     % for i = 3:nworms
    


%-----------------------------------------------------------
% ...for convenience, consider single valid's surrounded by "omega turns" as invalid
for i = 1 : nworms
    if i <= omega_range
        if valid(i) & all(~valid(1:i-1)) & all(~valid(i+1:i+omega_range))
            valid(i) = 0;
        end
    elseif i > (nworms-omega_range)
        if valid(i) & all(~valid(i+1:end)) & all(~valid(i-omega_range:i-1))
            valid(i) = 0;
        end
    else
        if valid(i) & all(~valid(i+1:i+omega_range)) & all(~valid(i-omega_range:i-1))
            valid(i) = 0;
        end
    end
end
        
%-----------------------------------------------------------

% Screen for omega turns; ensure that worms go in and come out head first
%----------------
% Find omega_exits (starts of data ranges-- where worm comes OUT of an omega turn)
omega_exits  = [];  % Exits: first valid points after alleged omega turns
                    %   (or at start of whole data set)
i = 1;  % Initialize counter
while isempty(omega_exits) & i <= omega_range      % Find first valid frame (if any)
                                                % within omega_range
    if valid(i) 
        omega_exits = i;
    end
    i = i+1;
end

for i = omega_range+1 : nworms
    if valid(i) & all(~valid(i - omega_range : i-1))
        omega_exits = [omega_exits i];
    end     % if valid
end     % for i = omega_range+1
    

%----------------
% Find omega_enters (ends of data ranges-- where worm curls INTO an omega turn)
omega_enters  = [];  % Enters: last valid points before alleged omega turns
                    %   (or at end of whole data set)
% Work backwards from the end of the data set, then reverse order...
i = nworms;  % Initialize counter
while isempty(omega_enters) & i > (nworms - omega_range) % Find last valid frame (if any)
                                                % within omega_range of end of data set
    if valid(i) 
        omega_enters = i;
    end
    i = i-1;
end

for i = (nworms - omega_range)  : -1 : 1
    if valid(i) & all(~valid(i+1 : i + omega_range))
        omega_enters = [omega_enters i];
    end     % if valid
end     % for i = (nworms - omega_range)

% Reverse order:
omega_enters = omega_enters(end:-1:1);
%----------------


% Sanity check:  (#enters should equal #exits!)
if prod(size(omega_enters)) ~= prod(size(omega_exits))
    error('Dude, somthing seriously wrong with your segmenting code!  ;-)')
end


% Index segments
segments = size(omega_enters,2);

% Create foraging vector:
end_1_foraging = [];
for i = 1:nworms-1
    % Identify which end of worm is foraging
%     head_foraging(i) = ( (x(i,1:3)  -x(i+1,1:3))  .^2 + (y(i,1:3)  -y(i+1,1:3)).^2 ) ...
%                      >= ( (x(i,end-2:end)-x(i+1,end-2:end)).^2 + (y(i,end-2:end)-y(i+1,end-2:end)).^2 );
    head_foraging(i) = ( (x(i,1)  -x(i+1,1))  .^2 + (y(i,1)  -y(i+1,1)).^2 ) ...
                    >= ( (x(i,end)-x(i+1,end)).^2 + (y(i,end)-y(i+1,end)).^2 );
end     % for i = 1:nworms-1
head_foraging(i+1) = head_foraging(i);

% Step through segments identifying whether segment is correctly oriented
%   based on foraging behavior (i.e. tip of head should move more than tip 
%   of tail)
forage = []; 
for i = 1:segments
    forage = [forage sum(head_foraging(omega_exits(i):omega_enters(i)))./(omega_enters(i)-omega_exits(i))];
end;    % for i = 1:size(segments,2)
forage = forage >= 0.5;     % Vector indicating if segments are CORRECTLY
                            %   oriented based on foraging (i.e. head moved
                            %   more than tail).

% Re-orient segments as necessary:
for i = 1:segments
    if forage(i) == 0
        x( omega_exits(i) : omega_enters(i) , :) ...
      = x( omega_exits(i) : omega_enters(i), end:-1:1);       % Reorder worms
  
        y( omega_exits(i) : omega_enters(i) , :) ...
      = y( omega_exits(i) : omega_enters(i), end:-1:1); 
    end     % if forage(i) == 0
end;    % for i = 1:size(segments,2)


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%% from the old AUTOPROC... %%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% Here's where the fun begins...  
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Mise en place:
accel_threshold = 10;         % Acceleration threshold (pixels/frame/frame)


% now compensate for table motion  
cx = mean(x')';     % Vector - Center of worm for each frame
cy = mean(y')';     % (length = #frames) ("Relative" center positions)
        
cxp = diff(cx);     % Vector - Movement between adjacent frames
cyp = diff(cy);     % (length = #frames - 1)
        
disp = sqrt(cxp.^2 + cyp.^2);   % Vector - Absolute value of movement 
                                % between adjacent frames
                                % (length = #frame - 1)

accx = diff(cxp);   % Acceleration = delta(velocities)
accy = diff(cyp);   %   (where velocity = displacement/frame)
                    %   so, Accel'n = rate of change of displacements

accx = [0; accx];   % Add 0th element to accelerations to line up 
accy = [0; accy];   % i'th element of acc* with i'th element of c*p.

val = (disp<30) & (~isnan(disp)) & ...
    (abs(accx) < accel_threshold) & (abs(accy) < accel_threshold);
                                    % Vector (l=#frames-1)
                                    % indicating valid for
                                    % movement less than 30 and
                                    % ~NaN (i.e. long-enough worms)
                                    % and no wild accelerations 
                                    % (i.e. microscope stage moves)
% val = (disp<30) & (~isnan(disp));   % Vector (l=#frames-1)
%                                     % indicating valid for
%                                     % movement less than 30 and
%                                     % ~NaN (i.e. long-enough worms)

nmoves = prod(size(val));       % Scalar  nmoves = #frames-1
% (WAS:)
% nworms = prod(size(val));       % Scalar  nworms = #frames-1
%                                 % Effectively, # moves


movenum = [1:nmoves];           % Index of moves 1 for move 1-->2,...
% (WAS:)
% wormnum = [1:nworms];           % Index of moves 1 for move 1-->2,...
        
cxp = cxp(logical(val),:);      % Eliminates all garbage (NaN's and
cyp = cyp(logical(val),:);      % big movements) from movement vectors...
        
movenum = movenum(logical(val));    % ...and also eliminates the 
                                    % corresponding index numbers...
% (WAS:)
% wormnum = wormnum(logical(val));    % ...and also eliminates the 
%                                     % corresponding index numbers...
        
nmoves = max(movenum) - min(movenum) + 1;   % ...and sets us up to trim off
                                            % invalid data at the beginning
                                            % and end of the data set 
                                            % (frames)...
% (WAS:)
% nworms = max(wormnum) - min(wormnum) + 1;   % ...and sets us up to trim off
%                                             % invalid data at the beginning
%                                             % and end of the data set 
%                                             % (frames)...

        
                        % ...Now interpolates between missing (removed)
                        % moves...
cxp = interp1(movenum, cxp, [min(movenum):max(movenum)], 'linear');
cyp = interp1(movenum, cyp, [min(movenum):max(movenum)], 'linear');
% (WAS:)
% cxp = interp1(wormnum, cxp, [min(wormnum):max(wormnum)], 'linear');
% cyp = interp1(wormnum, cyp, [min(wormnum):max(wormnum)], 'linear');
        
cxn = [0 cumsum(cxp)];  % ...and sets up a new pair of vectors that 
cyn = [0 cumsum(cyp)];  % should re-create the original x & y center 
                        % positions, (i.e. "virtual global" center x&y's)
                        % but without the garbage at the beginning and 
                        % end, and with the invalid data points in the 
                        % middle filled in...
        
nworms = nmoves + 1;    % The number of frames, WITHOUT the invalid
                        % frames at the beginning and end...
                                % Includes the interpolated frames in
                                % the middle
% (WAS:)
% nworms = nworms + 1;    % The number of frames, WITHOUT the invalid
%                         % frames at the beginning and end...
%                                 % Includes the interpolated frames in
%                                 % the middle
        
cxn = (cxn - mean(cxn))';   % This centers the new "virtual global" center
cyn = (cyn - mean(cyn))';   % x & y position vectors about their means
        
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% This is the part that I was confused on.....  
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
         
cx = cx(min(movenum):(max(movenum)+1),:);   % Takes the valid range of
cy = cy(min(movenum):(max(movenum)+1),:);   % (uncorrected) "Relative" x&y 
                                            % center positions (eliminating 
                                            % the garbage at the beginning  
                                            % and end) 
% (WAS:)
% cx = cx(1:nworms,:);    % Takes the FIRST nworms of cx & cy data...
% cy = cy(1:nworms,:);    % but it looked like we tried to eliminate 
%                         % the invalid data at the beginning
%                         % going from [min(wormnum):max(wormnum)]...
%                         % Now we take back the garbage at the beginning
%                         % AND the garbage in the middle too!
% % Should this be something like 
%         % cx = cx(min(wormnum):max(wormnum)); ?

                        
offx = cxn - cx;        % (Virtual global center positions) minus 
offy = cyn - cy;        % (Relative x&y center positions) =
                        % (Offsets necessary to convert Relative x&y 
                        % positions into Virtual Global positions)

% now do the correction (from Relative positions into Virtual Global 
%  positions)
x = x(min(movenum):(max(movenum)+1), :) + offx*ones(1,size(x,2)); 
y = y(min(movenum):(max(movenum)+1), :) + offy*ones(1,size(y,2)); 
% x = x(min(movenum):(max(movenum)+1), :) + offx*ones(1,13); 
% y = y(min(movenum):(max(movenum)+1), :) + offy*ones(1,13); 
% (WAS:)
% x = x(1:nworms,:) + offx*ones(1,13);    % ...or this
% y = y(1:nworms,:) + offy*ones(1,13);    % ...same question of the using the first *nworms*
%                                         % as opposed to the min(wormnum):max(wormnum)...

% But x & y still contain NaN's in the middle...  So,
% Interpolate over SINGLE NaN's:
for i = 2:(size(x,1)-1)
    if isnan(x(i,1))
        x(i,:) = interp1([i-1 ; i+1], [x(i-1,:) ; x(i+1,:)], i, 'linear');
        y(i,:) = interp1([i-1 ; i+1], [y(i-1,:) ; y(i+1,:)], i, 'linear');
    end
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%% from AUTOPROC %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


%-----------------------------------------------------
% ---> FROM autoproc_development_notes_5-27-03 <---
%-----------------------------------------------------
% enda = sqrt(diff(x(:,1)).^2 + diff(y(:,1)).^2);   % Vector of individual end-a displacements. 
% 
% endb = sqrt(diff(x(:,end)).^2 + diff(y(:,end)).^2);   % Vector of individual end-b displacements. 
% 
% total_enda = nansum(enda);   % Vector of total end-a displacement. 
% 
% total_endb = nansum(endb);   % Vector of total end-b displacement. 
%-----------------------------------------------------


% % % %-----------------------------------------------
% % %                 if omega_exits(k) - omega_enters(k) < 1
% % %                      (or omega_exits(k) == omega_enters(k)???)
% % %                     
% % %                     DISREGARD because 
% % %                        - snapshot
% % %                        - can't calculate foraging
% % % %-----------------------------------------------
% % % --> Calculate segment's foraging
% % % --> check curl at entry and exit


%----------------------------------------------------
%----------------------------------------------------
%----------------------------------------------------

% Philosophy change:  subsample in MANPROC...  -CJC 7-21-03
% % % % SUBSAMPLING:
% % % index = [1:size(x,1)]';
% % % if subsamplerate ~= 1
% % %     
% % % % ...ON SECOND THOUGHT, NEVER OVERWRITE YOUR ORIGINAL DATA...
% % % %     % save backup copy of original points file:
% % % %     [success,result] = dos(['copy ' directory '\points ' directory '\ORIGINALpoints']);
% % %     
% % % 
% % %     % extract every "subsamplerate'th" data point from 
% % %     %   x and y vectors
% % %     x       = x(1:subsamplerate:end,:);
% % %     y       = y(1:subsamplerate:end,:);
% % %     index   = index(1:subsamplerate:end);
% % %     
% % % % DON'T WASTE EFFORT ON WRITING NEW "POINTS" FILE
% % % % ---> JUST RETURN SUBSAMPLED x AND y!
% % % %     clear points
% % % %     
% % % %     % Block out memory for subsampled points matrix
% % % %     points = NaN * zeros(size(x,1)*2, size(x,2));
% % % % 
% % % %     % Fill in new points matrix:
% % % %     for i = 1:size(x,1)
% % % %         points(i*2-1, :)    = x(i,:);
% % % %         points(i*2, :)      = y(i,:);
% % % %     end     % for i = 1:size(xsub,1)
% % % %     
% % % %     % Save copy to disk
% % % %     save([directory '\points'], 'points', '-ascii');
% % %     
% % % end     % if subsamplerate ~= 1






